#!/usr/bin/env python
# -*- coding: utf-8 -*-

from os.path import join
from datetime import timedelta

import luigi
from yt.wrapper import with_context

from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib.socdem_helpers.inference_utils.voting import run_voting
from crypta.profile.utils import utils
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.utils.luigi_utils import BaseYtTask, YtTarget, ExternalInput, OldNodesByNameCleaner
from crypta.profile.tasks.common.merge_socdem_storage import JoinSocdemStorage
from crypta.profile.runners.export_profiles.lib.profiles_generation.merge_socdem_profiles import MergeBasicProfiles


class GetSocdemByCryptaId(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'yandexuid_classification_for_14days': MergeBasicProfiles(self.date, 'yandexuid'),
            'devid_classification_for_35days': MergeBasicProfiles(self.date, 'devid'),
            'socdem_storage': JoinSocdemStorage(self.date),
            'profiles_for_14days': ExternalInput(config.YANDEXUID_EXPORT_PROFILES_14_DAYS_TABLE),
        }

    def output(self):
        return {
            'crypta_id_profiles': YtTarget(join(config.CRYPTA_ID_PROFILES_YT_DIRECTORY, self.date)),
            'corrected_yandexuid_profiles': YtTarget(join(config.CORRECTED_YANDEXUID_PROFILES_YT_DIRECTORY, self.date)),
            'corrected_devid_profiles': YtTarget(join(config.CORRECTED_DEVID_PROFILES_YT_DIRECTORY, self.date)),
            'corrected_not_active_yandexuid_profiles': YtTarget(join(
                config.CORRECTED_NOT_ACTIVE_YANDEXUID_PROFILES_YT_DIRECTORY,
                self.date,
            )),
        }

    def run(self):
        with TimeTracker(self.__class__.__name__):
            run_voting(
                yt_client=self.yt,
                yql_client=self.yql.yql_client,
                yandexuid_14days_raw_classification=self.input()['yandexuid_classification_for_14days'].table,
                devid_35days_raw_classification_table=self.input()['devid_classification_for_35days'].table,
                thresholds=utils.get_socdem_thresholds_from_api(),
                date=self.date,
            )

            for table_name in self.output():
                yt_helpers.set_ttl(
                    table=self.output()[table_name].table,
                    ttl_timedelta=timedelta(days=14),
                    yt_client=self.yt,
                )


@with_context
def join_not_active_profiles(key, rows, context):
    socdem_row = None
    other_info_row = None

    for row in rows:
        if context.table_index == 0:
            socdem_row = row
        else:
            other_info_row = row

    if socdem_row and other_info_row:
        del socdem_row['update_time']
        other_info_row.update(socdem_row)
        other_info_row['update_time'] += 10
        yield other_info_row


class PrepareNotFreshProfilesExport(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'socdem': GetSocdemByCryptaId(self.date),
            'profiles_for_14days': ExternalInput(config.YANDEXUID_EXPORT_PROFILES_14_DAYS_TABLE),
            'cleaner': OldNodesByNameCleaner(
                self.date,
                folder=config.CORRECTED_NOT_ACTIVE_YANDEXUID_PROFILES_WITH_SEGMENTS_YT_DIRECTORY,
                lifetime=config.NUMBER_OF_INTERMEDIATE_PROFILES_TABLES_TO_KEEP,
            ),
        }

    def output(self):
        return YtTarget(join(config.CORRECTED_NOT_ACTIVE_YANDEXUID_PROFILES_WITH_SEGMENTS_YT_DIRECTORY, self.date))

    def run(self):
        with TimeTracker(self.__class__.__name__):
            with self.yt.Transaction():
                self.yt.create_empty_table(
                    self.output().table,
                    schema=utils.crypta_yandexuid_profiles_schema,
                )

                self.yt.run_reduce(
                    join_not_active_profiles,
                    [self.input()['socdem']['corrected_not_active_yandexuid_profiles'].table,
                     self.input()['profiles_for_14days'].table],
                    self.output().table,
                    reduce_by='yandexuid',
                )

                self.yt.run_sort(
                    self.output().table,
                    sort_by='yandexuid',
                )
