#!/usr/bin/env python
# -*- coding: utf-8 -*-
import json
import os

import luigi

import crypta.lib.python.spine.consts.attributes as spine_attributes
import crypta.lib.python.yt.time_utils as yt_time_utils
from crypta.profile.lib import date_helpers
from crypta.profile.lib import profile_helpers
from crypta.profile.runners.export_profiles.lib.profiles_generation.add_segments_to_profiles import AddSegmentsToProfiles
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_socdem_by_crypta_id import PrepareNotFreshProfilesExport
from crypta.profile.runners.export_profiles.lib.profiles_generation.postprocess_profiles import (
    get_export_expressions,
    PostprocessProfilesMapper,
)
from crypta.profile.utils import (
    interests_helpers,
    reactor,
    utils,
)
from crypta.profile.utils.config import config
from crypta.profile.utils.config.secrets import get_secrets
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.utils.luigi_utils import (
    YtTarget,
    BaseYtTask,
)
from crypta.profile.utils.segment_utils.boolparser import ExpressionParser


def report_to_step(date_string):
    from library.python.step.client import Client, DEFAULT_HOSTS

    statbox_step_client = Client(
        host=DEFAULT_HOSTS[config.environment],
        token=get_secrets().get_secret('STATBOX_STEP_OAUTH_TOKEN'),
    )
    statbox_step_arguments = {
        'type': 'crypta',
        'group': 'crypta-yandexuid-profiles-log',
        'scale': '1d',
        'path': '//statbox/crypta-yandexuid-profiles-log/{date_string}'.format(date_string=date_string),
        'timestamp': '{date_string}T00:00:00'.format(date_string=date_string),
        'cluster': 'hahn',
    }
    statbox_step_client.create_event(name='cluster_table_publish', params=statbox_step_arguments)


SECONDS_IN_A_DAY = 24 * 60 * 60


class GetStatboxCryptaProfilesLog(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'active_profiles': AddSegmentsToProfiles(self.date),
            'not_active_profiles': PrepareNotFreshProfilesExport(self.date),
        }

    def output(self):
        yesterday = date_helpers.get_yesterday(self.date)
        return YtTarget(os.path.join(config.YANDEXUID_PROFILES_EXPORT_DIRECTORY, yesterday))

    def run(self):
        with TimeTracker(self.__class__.__name__):

            # direct conformity with BB keywords
            bb_keyword_id_to_fields = {str(key): value for key, value in utils.bb_keyword_id_to_field_name.iteritems()}
            del bb_keyword_id_to_fields['217']
            # indirect connection with BB keywords
            not_strict_bb_keyword_id_to_field_names = {
                '198': 'top_common_sites',
                '595': 'affinitive_sites',
                '217': {
                    'probabilistic_segments': 'default',
                    'interests_composite': list(profile_helpers.interests_composite_segment_ids),
                },
            }

            output_table = self.output().table

            with self.yt.Transaction():
                self.yt.create_empty_table(
                    output_table,
                    schema=utils.crypta_yandexuid_profiles_schema,
                    additional_attributes={
                        'bb_keywords_to_field_name': bb_keyword_id_to_fields,
                        'not_strict_bb_keyword_id_to_field_names': not_strict_bb_keyword_id_to_field_names,
                    },
                )

                exports_expressions = get_export_expressions(interests_helpers.LabSegmentsInfo().segments)
                parser = ExpressionParser(exports_expressions, self.logger)
                trees = parser.build_trees()

                current_time = date_helpers.from_utc_date_string_to_noon_timestamp(self.date)
                outdated_shortterm_interests_threshold = current_time - SECONDS_IN_A_DAY * config.SHORTTERM_INTERESTS_EXPIRE_DAYS

                operation = self.yt.run_map(
                    PostprocessProfilesMapper(
                        trees=trees,
                        exports_expressions=exports_expressions,
                        outdated_shortterm_interests_threshold=outdated_shortterm_interests_threshold,
                    ),
                    source_table=[
                        self.input()['active_profiles'].table,
                        self.input()['not_active_profiles'].table,
                    ],
                    destination_table=output_table,
                    spec={'data_size_per_job': config.POSTPROCESS_PROFILES_MAPPER_DATA_SIZE_PER_JOB},
                )
                self.logger.info(json.dumps(operation.get_job_statistics()))
                self.yt.sort_if_needed(output_table, sort_by='yandexuid')
                self.yt.set_attribute(output_table, spine_attributes.GENERATE_DATETIME, yt_time_utils.get_current_yt_timestamp())

                yesterday = date_helpers.get_yesterday(self.date)
                if config.environment == 'production':
                    report_to_step(date_string=yesterday)


class ReportYandexuidProfilesLogCreation(reactor.ReportProfilesLogCreation):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ReportYandexuidProfilesLogCreation, self).__init__(*args, **kwargs)
        self.namespace_path = '/crypta/profile/hahn/crypta-yandexuid-profiles-log'

    def requires(self):
        return GetStatboxCryptaProfilesLog(self.date)
