#!/usr/bin/env python
# -*- coding: utf-8 -*-

import collections

from crypta.profile.lib import profile_helpers
from crypta.profile.lib.socdem_helpers import socdem_config
from crypta.profile.utils.segment_storage import DICT_FIELDS, LIST_FIELDS
from crypta.profile.utils.segment_utils import boolparser
from crypta.profile.utils.utils import (
    bb_keyword_id_to_field_name,
)


Export = collections.namedtuple("Export", ['field_name', 'segment_id', 'expressions'])


class Profile(object):
    def __init__(self, row):
        self.profile = row

        for field in DICT_FIELDS:
            if self.profile.get(field) is None:
                self.profile[field] = {}

        for field in LIST_FIELDS:
            if self.profile.get(field) is None:
                self.profile[field] = set()
            else:
                self.profile[field] = set(self.profile[field])

        if self.profile.get('exact_socdem') is None:
            self.profile['exact_socdem'] = {}

    def __contains__(self, item):
        return self.profile.__contains__(item)

    def __getitem__(self, field_name):
        return self.profile[field_name]

    def __setitem__(self, field_name, value):
        self.profile[field_name] = value

    def get_gender(self):
        return self.profile['exact_socdem'].get('gender')

    def get_age_segment(self):
        return self.profile['exact_socdem'].get('age_segment')

    def get_income_segment(self):
        return self.profile['exact_socdem'].get('income_segment')

    def get_income_5_segment(self):
        return self.profile['exact_socdem'].get('income_5_segment')

    def get_record(self):
        for field in DICT_FIELDS:
            if not self.profile[field]:
                self.profile[field] = None

        for field in LIST_FIELDS:
            if not self.profile[field]:
                self.profile[field] = None
            else:
                self.profile[field] = list(self.profile[field])

        if not self.profile['exact_socdem']:
            self.profile['exact_socdem'] = None

        return self.profile

    def set_probabilistic_subsegment(self, segment, subsegment='0', value=1.0):
        self.profile['probabilistic_segments'].setdefault(segment, {})[subsegment] = value

    def remove_probabilistic_subsegment(self, segment, subsegment='0'):
        if self.profile['probabilistic_segments']:
            if segment in self.profile['probabilistic_segments']:
                if subsegment in self.profile['probabilistic_segments'][segment]:
                    del self.profile['probabilistic_segments'][segment][subsegment]
                if not self.profile['probabilistic_segments'][segment]:
                    del self.profile['probabilistic_segments'][segment]

    def get_segment(self, field_name, segment_id):
        if field_name == 'probabilistic_segments':
            return self.profile[field_name].get(str(segment_id), {}).get('0')
        if field_name == 'interests_composite':
            return self.profile[field_name].get(str(segment_id), {}).get('1')
        if field_name in ('heuristic_segments', 'marketing_segments', 'lal_common', 'lal_private', 'lal_internal'):
            return self.profile[field_name].get(str(segment_id))
        if field_name in ('heuristic_common', 'heuristic_private', 'heuristic_internal', 'longterm_interests', 'audience_segments'):
            if segment_id in self.profile[field_name]:
                return 1
        if field_name == 'shortterm_interests':
            if str(segment_id) in self.profile[field_name]:
                return self.profile[field_name].get(str(segment_id))

        return None

    def set_segment(self, field_name, segment_id, value):
        if field_name == 'probabilistic_segments':
            self.profile[field_name].setdefault(str(segment_id), {})['0'] = value
        elif field_name == 'interests_composite':
            self.profile[field_name].setdefault(str(segment_id), {})['1'] = value
            self.profile[field_name].setdefault(str(segment_id), {})['0'] = 1 - value
        elif field_name == 'heuristic_segments':
            self.profile[field_name][str(segment_id)] = 1
        elif field_name in ('marketing_segments', 'lal_common', 'lal_private', 'lal_internal'):
            self.profile[field_name][str(segment_id)] = value
        elif field_name == 'shortterm_interests':
            self.profile[field_name][str(segment_id)] = value
        elif field_name in (
            'heuristic_common', 'heuristic_private', 'heuristic_internal',
            'longterm_interests', 'audience_segments',
        ):
            self.profile[field_name].add(int(segment_id))


def _get_segment_field_name(keyword_id, segment_id):
    if str(keyword_id) == '217':
        if str(segment_id) in profile_helpers.interests_composite_segment_ids:
            return 'interests_composite'
        else:
            return 'probabilistic_segments'
    else:
        return bb_keyword_id_to_field_name.get(keyword_id)


def get_export_expression(export):
    return Export(
        field_name=_get_segment_field_name(export.keywordId, export.segmentId),
        segment_id=export.segmentId,
        expressions=u' OR '.join([expression.normalized for expression in export.fullExpressions]) if export.fullExpressions else export.id,
    )


def get_export_expressions(segments):
    return {
        export.id: get_export_expression(export)
        for segment in segments
        for export in segment.exports.exports
        if export.state in ('CREATED', 'ACTIVE', 'DEPRECATED', 'DISABLE_REQUESTED') and not export.hasExpressionErrors
    }


def format_socdem_export(field_name, segment_id, profile):
    if field_name == 'user_age_6s':
        return profile.get_age_segment() == profile_helpers.six_segment_age_id_to_name[segment_id]
    elif field_name == 'income_5_segments':
        return profile.get_income_5_segment() == profile_helpers.five_segment_income_id_to_name[segment_id]
    else:
        return profile.get_gender() == profile_helpers.gender_id_to_name[segment_id]


class Evaluator(object):
    def __init__(self, profile, dummy_profile, exports_expressions, trees):
        self.profile = profile
        self.dummy_profile = dummy_profile
        self.exports_expressions = exports_expressions
        self.trees = trees
        self.checked_exports = {}
        self.profile_exports = {}

    def evaluate_root(self, export_id):
        if export_id in self.checked_exports:
            return self.checked_exports[export_id]

        root = self.trees[export_id]

        has_segment, value = root.evaluate(self, export_id)

        self.checked_exports[export_id] = has_segment, value
        if has_segment:
            export = self.exports_expressions[export_id]
            self.dummy_profile.set_segment(
                export.field_name,
                export.segment_id,
                value,
            )

        return has_segment, value

    def evaluate_independent(self, export_id, root_export_id):
        if export_id == root_export_id:
            return self.get_from_profile(export_id)
        else:
            return self.evaluate_root(export_id)

    def get_from_profile(self, export_id):
        if export_id in self.profile_exports:
            return self.profile_exports[export_id]

        export = self.exports_expressions[export_id]
        field_name = export.field_name
        segment_id = export.segment_id

        if field_name in socdem_config.yet_another_segment_names_by_label_type:
            has_segment = format_socdem_export(field_name, segment_id, self.profile)
        else:
            if field_name in DICT_FIELDS:
                segment_id = str(segment_id)
            has_segment = segment_id in self.profile[field_name]

        value = self.profile.get_segment(field_name, segment_id)
        self.profile_exports[export_id] = has_segment, value
        return has_segment, value


def get_reverse_dependencies(dependencies, self_copy_exports, exports_expressions, trees):
    reverse_deps = collections.defaultdict(set)
    socdem_exports = get_socdem_exports(exports_expressions)

    for export_id, children in dependencies.iteritems():
        if export_id in self_copy_exports:
            continue

        need_socdem = check_if_export_has_terms_with_socdem_only(trees, export_id, socdem_exports)

        for child_id in children:
            export = exports_expressions[child_id]
            key = make_key(export)
            if need_socdem or child_id not in socdem_exports:
                reverse_deps[key].add(export_id)

    return reverse_deps


def get_socdem_exports(exports_expressions):
    return {
        export_id
        for export_id, export in exports_expressions.iteritems()
        if export.field_name in socdem_config.yet_another_segment_names_by_label_type
    }


def check_if_export_has_terms_with_socdem_only(trees, export_id, socdem_exports):
    def has_terms_with_socdem_only(export_id_to_evaluate, root_export_id):
        if export_id_to_evaluate == root_export_id:
            return export_id_to_evaluate in socdem_exports
        else:
            return trees[export_id_to_evaluate].has_terms_with_socdem_only(has_terms_with_socdem_only, export_id_to_evaluate)

    return trees[export_id].has_terms_with_socdem_only(has_terms_with_socdem_only, export_id)


def get_negative_segments(deps):
    return {export_id for export_id, children in deps.iteritems() if not children}


def get_dependencies_and_self_copy_exports(trees):
    dependencies = dict()
    self_copy_exports = set()

    def get_full_dependencies(export_id, root_export_id):
        if export_id == root_export_id:
            return boolparser.Segments({export_id})
        else:
            return trees[export_id].get_full_dependencies(get_full_dependencies, export_id)

    for export_id, node in trees.iteritems():
        this_deps = node.get_full_dependencies(get_full_dependencies, export_id)
        dependencies[export_id] = this_deps.included

        if this_deps == boolparser.Segments({export_id}):
            self_copy_exports.add(export_id)

    return dependencies, self_copy_exports


def make_key(export):
    if export.field_name in DICT_FIELDS:
        return export.field_name, str(export.segment_id)
    elif export.field_name in LIST_FIELDS:
        return export.field_name, export.segment_id
    elif export.field_name in socdem_config.yet_another_segment_names_by_label_type:
        if export.field_name == 'user_age_6s':
            return 'age_segment', profile_helpers.six_segment_age_id_to_name[export.segment_id]
        elif export.field_name == 'income_5_segments':
            return 'income_5_segment', profile_helpers.five_segment_income_id_to_name[export.segment_id]
        else:
            return 'gender', profile_helpers.gender_id_to_name[export.segment_id]


class PostprocessProfilesMapper(object):
    def __init__(self, trees, exports_expressions, outdated_shortterm_interests_threshold):
        self.trees = trees
        self.exports_expressions = exports_expressions
        self.outdated_shortterm_interests_threshold = outdated_shortterm_interests_threshold

        dependencies, self_copy_exports = get_dependencies_and_self_copy_exports(self.trees)

        self.reverse_dependencies = get_reverse_dependencies(dependencies, self_copy_exports, exports_expressions, trees)
        self.self_copy_segments = {make_key(exports_expressions[export_id]) for export_id in self_copy_exports}
        self.negative_segments = get_negative_segments(dependencies)

    def apply_expressions(self, profile):
        expressions_to_evaluate = set(self.negative_segments)
        dummy_profile = Profile({})

        for field in LIST_FIELDS:
            to_copy = set()
            for segment_id in profile[field]:
                expressions_to_evaluate.update(self.reverse_dependencies[(field, segment_id)])
                if (field, segment_id) in self.self_copy_segments:
                    to_copy.add(segment_id)

            dummy_profile[field] = to_copy

        for field in DICT_FIELDS:
            to_copy = dict()
            for segment_id, value in profile[field].iteritems():
                expressions_to_evaluate.update(self.reverse_dependencies[(field, segment_id)])
                if (field, segment_id) in self.self_copy_segments:
                    to_copy[segment_id] = value

            dummy_profile[field] = to_copy

        for field, segment_id in profile['exact_socdem'].iteritems():
            expressions_to_evaluate.update(self.reverse_dependencies[(field, segment_id)])

        evaluator = Evaluator(profile, dummy_profile, self.exports_expressions, self.trees)
        for export_id in expressions_to_evaluate:
            evaluator.evaluate_root(export_id)

        for field in LIST_FIELDS | DICT_FIELDS:
            profile[field] = dummy_profile[field]

        return profile

    @staticmethod
    def delete_source_from_lal(profile):
        if 1404 in profile['heuristic_internal']:
            if '1407' in profile['lal_internal']:
                del profile['lal_internal']['1407']
            profile['heuristic_internal'].remove(1404)

        if 1405 in profile['heuristic_internal']:
            if '1409' in profile['lal_internal']:
                del profile['lal_internal']['1409']
            profile['heuristic_internal'].remove(1405)

        if 1406 in profile['heuristic_internal']:
            if '1408' in profile['lal_internal']:
                del profile['lal_internal']['1408']
            profile['heuristic_internal'].remove(1406)

    def delete_outdated_shortterm_interests(self, profile):
        profile['shortterm_interests'] = {
            interest_id: interest_timestamp
            for interest_id, interest_timestamp in profile['shortterm_interests'].iteritems()
            if interest_timestamp > self.outdated_shortterm_interests_threshold
        }

    def apply_all_rules(self, row):
        profile = Profile(row)
        self.delete_source_from_lal(profile)
        profile = self.apply_expressions(profile)
        self.delete_outdated_shortterm_interests(profile)
        return profile.get_record()

    def __call__(self, row):
        yield self.apply_all_rules(row)
