#!/usr/bin/env python
# -*- coding: utf-8 -*-

import abc
import os

import luigi

from crypta.lib.python import time_utils
from crypta.profile.lib import date_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import BaseYtTask, YtTarget


class InterestsLogProcessor(BaseYtTask):
    date = luigi.Parameter()
    log_name = luigi.Parameter()
    log_path = luigi.Parameter()

    @property
    def processed_table(self):
        return os.path.join(
            config.INTERESTS_PROCESSED_FOLDER,
            self.date,
            self.log_name,
            os.path.basename(self.log_path),
        )

    @property
    def to_bigb_raw_table(self):
        return os.path.join(
            config.INTERESTS_TO_BIGB_RAW_FOLDER,
            '%s_%s' % (self.log_name, os.path.basename(self.log_path)),
        )

    def output(self):
        return YtTarget(self.processed_table)

    def run(self):
        self.yt.config['spec_defaults']['pool'] = config.INTERESTS_POOL
        with self.yt.Transaction() as transaction:
            self.create_empty_processed_table()
            self.fill_processed_table(transaction)
            self.sort_processed_table()
            self.add_generate_datetime_attribute(self.processed_table)
            self.logger.info('Finished %s table processing.' % self.log_path)

    def create_empty_processed_table(self):
        self.yt.create_empty_table(
            self.processed_table,
            schema={
                'yandexuid': 'uint64',
                'timestamp': 'uint64',
                'date': 'string',
                'datetime_bin': 'string',
                'log_name': 'string',
                'interest_lab_id': 'string',
                'raw_data': 'any',
                'data': 'any',
                'data_source': 'string',
                'data_type': 'string',
            },
        )

    def create_empty_to_bigb_raw_table(self):
        self.yt.create_empty_table(
            self.to_bigb_raw_table,
            schema={
                'yandexuid': 'uint64',
                'shortterm_interests': 'any',
            },
        )

    def move_to_bigb_table(self):
        if self.yt.exists(self.to_bigb_raw_table):
            if self.yt.row_count(self.to_bigb_raw_table) == 0:
                self.yt.remove(self.to_bigb_raw_table)
                self.logger.warning('%s (to_bigb_raw_table) is empty.' % self.to_bigb_raw_table)
            else:
                self.yt.create_directory(config.INTERESTS_TO_BIGB_FOLDER)
                self.yt.run_sort(
                    self.to_bigb_raw_table,
                    os.path.join(
                        config.INTERESTS_TO_BIGB_FOLDER,
                        str(time_utils.get_current_time()),
                    ),
                    sort_by='yandexuid',
                )
                self.yt.remove(self.to_bigb_raw_table)
        else:
            self.logger.warning('%s (to_bigb_raw_table) doesn\'t exist.' % self.to_bigb_raw_table)

    @abc.abstractmethod
    def fill_processed_table(self, transaction):
        pass

    def sort_processed_table(self):
        self.yt.sort_if_needed(
            self.processed_table,
            sort_by=[
                'yandexuid',
                'interest_lab_id',
                'date',
                'datetime_bin',
            ],
        )

    def add_generate_datetime_attribute(self, table):
        self.yt.set_attribute(
            table,
            'generate_datetime',
            date_helpers.to_date_string(
                time_utils.get_current_moscow_datetime(),
                date_format=date_helpers.YT_DATETIME_FORMAT,
            ),
        )
