#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import datetime
import os

import luigi

from crypta.profile.lib import date_helpers
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.runners.interests.lib.data_preparation import PrepareCatalogiaMappingForBB
from crypta.profile.runners.interests.lib.log_processors.bigb_dump import BBDumpProcessor
from crypta.profile.runners.interests.lib.log_processors.metrics import MetricsProcessor
from crypta.profile.runners.interests.lib.log_processors.reqans import ReqansProcessor
from crypta.profile.runners.interests.lib.longterm_builder import LongtermInterestsTask
from crypta.profile.runners.interests.lib.shortterm_builder import ShorttermInterestsTask
from crypta.profile.utils.config import config
from crypta.profile.utils.interests_helpers import get_catalogia_rules_for_interests
from crypta.profile.utils.luigi_utils import OldNodesByNameCleaner
from crypta.profile.utils.yt_utils import get_yt_client

log_processors = {
    'metrics': MetricsProcessor,
    'reqans': ReqansProcessor,
}


class InterestsLogProcessorTask(luigi.WrapperTask):
    log_name = luigi.Parameter()
    log_folder = luigi.Parameter()
    max_datetime_isoformat = luigi.Parameter()
    max_number_of_tables = luigi.Parameter(default=None, significant=False)

    def requires(self):
        required_tasks = []

        yt = get_yt_client()

        tables = sorted(yt.list(self.log_folder), reverse=True)
        number_of_tables = self.max_number_of_tables or len(tables)

        for priority, log_tablename in enumerate(tables[:number_of_tables], start=1):
            log_path = os.path.join(self.log_folder, log_tablename)

            if yt.get_attribute(log_path, 'creation_time') > self.max_datetime_isoformat:
                continue

            log_date = log_tablename[:10]
            new_task = log_processors[self.log_name](
                date=log_date,
                log_name=self.log_name,
                log_path=log_path,
            )
            if not new_task.complete():
                new_task.priority = priority
                yt.create_directory(os.path.dirname(new_task.processed_table))
                yt.create_directory(os.path.dirname(new_task.to_bigb_raw_table))

                required_tasks.append(new_task)

        return required_tasks


class ProcessBBDumps(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        catalogia_category_to_rule_revision_ids, rule_revision_id_to_segment_ids = get_catalogia_rules_for_interests()

        required_tasks = []
        for date in date_helpers.generate_back_dates(self.date, config.NUMBER_OF_DAYS_TO_CALCULATE_RULES):
            required_tasks.append(BBDumpProcessor(
                date=date,
                catalogia_category_to_rule_revision_ids=FrozenDict(catalogia_category_to_rule_revision_ids),
                rule_revision_id_to_segment_ids=FrozenDict(rule_revision_id_to_segment_ids),
            ))

        return required_tasks


class InterestsLogProcessorsTask(luigi.WrapperTask):
    date = luigi.Parameter(default=str(datetime.date.today()))
    max_datetime_isoformat = datetime.datetime.isoformat(datetime.datetime.now())

    def requires(self):
        return [
            ProcessBBDumps(date_helpers.get_yesterday(str(self.date))),
            InterestsLogProcessorTask(
                log_name='reqans',
                log_folder='//home/logfeller/logs/search-proto-reqans-log/30min',
                max_datetime_isoformat=self.max_datetime_isoformat,
                max_number_of_tables=48,
            ),
            InterestsLogProcessorTask(
                log_name='metrics',
                log_folder=config.METRICS_PARSED_DIR,
                max_datetime_isoformat=self.max_datetime_isoformat,
                max_number_of_tables=3,
            ),
            OldNodesByNameCleaner(
                date=self.date,
                folder=config.INTERESTS_PROCESSED_FOLDER,
                lifetime=config.INTERESTS_PROCESSED_FOLDER_TTL_DAYS,
            ),
        ]


class InterestsTasks(luigi.WrapperTask):
    date = luigi.Parameter(default=str(datetime.date.today()))

    def requires(self):
        yesterday = date_helpers.get_yesterday(self.date)

        return [
            PrepareCatalogiaMappingForBB(self.date),
            LongtermInterestsTask(yesterday),
            ShorttermInterestsTask(yesterday),
        ]


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Run interests tasks')
    parser.add_argument('--task', dest='task', help='task class name', default='InterestsLogProcessorsRunner')
    args = parser.parse_args()

    luigi.run(
        [
            args.task,
            '--scheduler-url', config.LUIGI_SCHEDULER_URL,
            '--workers', '4',
        ]
    )
