"""
Crypta user profile LaL samples builder

Usage:
  main.py [--log-file-path=<path>] [--date=<date>]
  main.py (-h | --help)

Options:
  --log-file-path=<path>        Path for logs
  --date=<date>                 Date string
"""

import datetime
import logging
import os

import luigi
from docopt import docopt
import yt.logger as yt_logger

from crypta.profile.lib import (
    date_helpers,
    luigi_helpers,
)
from crypta.profile.utils.config import config


from crypta.profile.runners.lal_samples.lib.afisha_users import AfishaUsers
from crypta.profile.runners.lal_samples.lib.common_lals import CommonLals
from crypta.profile.runners.lal_samples.lib.interior_designers import InteriorDesigners
from crypta.profile.runners.lal_samples.lib.kinopoisk_online_users import KinopoiskOnlineUsers
from crypta.profile.runners.lal_samples.lib.new_settlers import NewSettlers
from crypta.profile.runners.lal_samples.lib.q_answering_users import QAnsweringUsers
from crypta.profile.runners.lal_samples.lib.switched_iphone_to_samsung import SwitchedIphoneToSamsung
from crypta.profile.runners.lal_samples.lib.ultima_taxi_users import UltimaTaxiUsers


NUMBER_OF_WORKERS = 2


class LalSamplesTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date
        yesterday = date_helpers.get_yesterday(today)

        if config.environment == 'testing':
            return [
                UltimaTaxiUsers(today),
            ]
        else:
            return [
                AfishaUsers(today),
                CommonLals(today),
                InteriorDesigners(yesterday),
                KinopoiskOnlineUsers(yesterday),
                NewSettlers(yesterday),
                QAnsweringUsers(today),
                SwitchedIphoneToSamsung(yesterday),
                UltimaTaxiUsers(today),
            ]


def main():
    arguments = docopt(__doc__)

    config.DEFAULT_POOL = config.SEGMENTS_POOL

    if arguments['--log-file-path']:
        config.LOCAL_LOGS_DIRECTORY = config.TASKS_LOGS_DIRECTORY = arguments['--log-file-path']
        config.LOCAL_STORAGE_DIRECTORY = arguments['--log-file-path']

    date = str(datetime.date.today())
    if arguments['--date']:
        date = arguments['--date']

    task_name = LalSamplesTask.__name__

    luigi_helpers.setup_external_tasks_retry()

    yt_logger.LOGGER.setLevel(logging.DEBUG)
    yt_logger.LOGGER.handlers = [logging.FileHandler(os.path.join(config.LOCAL_LOGS_DIRECTORY, "yt.log"))]

    luigi.run(
        [
            task_name,
            '--scheduler-url', config.LUIGI_SCHEDULER_URL,
            '--workers', str(NUMBER_OF_WORKERS),
            '--date', date,
            '--log-level', 'INFO',
        ],
    )


if __name__ == '__main__':
    main()
