#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInputDate
from crypta.profile.utils.segment_utils.builders import (
    LalSampleBuilder,
    LaLParams,
)


heuristic_keyword_id_to_condition_dict = {
    216: 'DictContains(Yson::ConvertToUint64Dict(heuristic_segments), "{segment_id}")',
    547: 'ListHas(Yson::ConvertToUint64List(heuristic_common), {segment_id})',
    549: 'ListHas(Yson::ConvertToUint64List(heuristic_internal), {segment_id})',
    557: 'ListHas(Yson::ConvertToUint64List(audience_segments), {segment_id})',
    601: 'ListHas(Yson::ConvertToUint64List(longterm_interests), {segment_id})',
}

indevice_matching_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    indevice_matching.yandexuid AS yandexuid,
    lal_sample.segment_name AS segment_name
FROM LIKE(`{common_lal_samples_dir}`, '%') AS lal_sample
INNER JOIN `{indevice_yandexuid_matching}` AS indevice_matching
USING (id, id_type)
GROUP BY indevice_matching.yandexuid, lal_sample.segment_name;
"""

intersection_with_heuristic_analogies_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    CAST(lal_sample.yandexuid AS String) AS yandexuid,
    lal_sample.segment_name AS segment_name
FROM `{samples_by_yuid_table}` AS lal_sample
INNER JOIN `{profiles_for_14days_table}` AS profiles
USING(yandexuid)
{conditions};
"""


class CommonLals(LalSampleBuilder):
    def requires(self):
        return ExternalInputDate(table=config.YANDEXUID_EXPORT_PROFILES_14_DAYS_TABLE, date=self.date)

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as samples_by_yuid:
            self.yql.query(
                indevice_matching_query.format(
                    common_lal_samples_dir=config.COMMON_LAL_SAMPLES_FOLDER,
                    indevice_yandexuid_matching=config.INDEVICE_YANDEXUID,
                    output_table=samples_by_yuid,
                ),
                transaction=self.transaction,
            )

            lals_params = list()
            samples_without_heuristic_similar = list()
            samples_with_heuristic_similar = dict()
            for table in self.yt.list(config.COMMON_LAL_SAMPLES_FOLDER, absolute=True):
                self.logger.info('Processing the table {}'.format(table))
                table_attributes = self.yt.get_table_attributes(table)

                segment_name = table_attributes['segment_name']
                assert isinstance(segment_name, basestring), 'attribute segment_name should be string type'

                segment_type = table_attributes['segment_type']
                assert isinstance(segment_type, basestring), 'attribute segment_type should be string type'
                assert segment_type in ('audience_segments', 'lal_internal', 'marketing_segments')

                lals_params.append(LaLParams(
                    name=segment_name,
                    id=int(table_attributes['segment_id']),
                    type=segment_type,
                    coverage=int(table_attributes['segment_coverage']),
                    include_input=bool(table_attributes['include_input']),
                ))

                heuristic_segment_keyword = table_attributes.get('heuristic_segment_keyword')
                heuristic_segment_id = table_attributes.get('heuristic_segment_id')
                if heuristic_segment_keyword is not None and heuristic_segment_id is not None:
                    assert heuristic_segment_keyword in heuristic_keyword_id_to_condition_dict
                    samples_with_heuristic_similar[segment_name] = {
                        'heuristic_segment_keyword': int(heuristic_segment_keyword),
                        'heuristic_segment_id': int(heuristic_segment_id),
                    }
                else:
                    samples_without_heuristic_similar.append(segment_name)

            conditions = 'WHERE segment_name in {}'.format(tuple(samples_without_heuristic_similar))
            for segment_name in samples_with_heuristic_similar:
                heuristic_segment_keyword = samples_with_heuristic_similar[segment_name]['heuristic_segment_keyword']
                heuristic_segment_id = samples_with_heuristic_similar[segment_name]['heuristic_segment_id']
                conditions += ' OR (segment_name == \'{segment_name}\' AND {condition})'.format(
                    segment_name=segment_name,
                    condition=heuristic_keyword_id_to_condition_dict[heuristic_segment_keyword].format(
                        segment_id=heuristic_segment_id,
                    ),
                )

            self.yql.query(
                intersection_with_heuristic_analogies_query.format(
                    samples_by_yuid_table=samples_by_yuid,
                    profiles_for_14days_table=inputs.table,
                    output_table=output_path,
                    conditions=conditions,
                ),
                transaction=self.transaction,
            )

            self.yt.set_attribute(
                output_path,
                'generate_date',
                self.date,
            )

            self.prepare_samples_for_lal(
                output_path,
                id_field='yandexuid',
                lals_params=lals_params,
            )
