#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import (
    LalSampleBuilder,
    LaLParams,
)
from crypta.profile.utils.utils import get_matching_table


afisha_users_query_template = """
$subscriptions = AsSet('basic-kinopoisk', 'kinopoisk-plus', 'kp-amediateka', 'station-lease-plus-kinopoisk');

$kino_and_amedia_puids = (
SELECT
    puid,
    end_date,
    start_date,
    state
FROM `{statuses}`
WHERE bundle IN $subscriptions
);

$bought_with_subscription = (
SELECT
    transactions.puid as puid
FROM `{kinopoisk_online_transactions}` as transactions
INNER JOIN $kino_and_amedia_puids as status
USING(puid)
WHERE status.start_date <= transactions.`date` AND transactions.`date` <= status.end_date
GROUP BY transactions.puid
);

$segment_crypra_id = (
SELECT
    matching.target_id as crypta_id,
FROM (
    SELECT
        transactions.puid as puid,
    FROM `{kinopoisk_online_transactions}` as transactions
    LEFT ONLY JOIN $bought_with_subscription as bought_with_subscription
    USING(puid)
) as kinopoisk
INNER JOIN `{matching_puid}` as matching
ON matching.id == kinopoisk.puid
GROUP BY matching.target_id
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    matching.id as yandexuid,
    'kinopoisk_online_users' as segment_name
FROM `{matching_yandexuid}` as matching
LEFT SEMI JOIN $segment_crypra_id as crypta_id
ON crypta_id.crypta_id == matching.target_id
"""

KINOPOISK_TRANSACTIONS_DIRECTORY = '//home/plus/services/kinopoisk/transactions/1d'


class KinopoiskOnlineUsers(LalSampleBuilder):
    def requires(self):
        return {
            'kinopoisk_online_transactions': ExternalInput(os.path.join(KINOPOISK_TRANSACTIONS_DIRECTORY, self.date)),
            'statuses': ExternalInput(os.path.join(config.SUBSCRIBERS_CUBE, self.date)),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            query_string=afisha_users_query_template.format(
                kinopoisk_online_transactions=inputs['kinopoisk_online_transactions'].table,
                statuses=inputs['statuses'].table,
                matching_yandexuid=get_matching_table('yandexuid', 'crypta_id'),
                matching_puid=get_matching_table('puid', 'crypta_id'),
                output_table=output_path,
            ),
            transaction=self.transaction,
        )

        self.yt.set_attribute(
            output_path,
            'generate_date',
            self.date
        )

        self.prepare_samples_for_lal(
            input_table=output_path,
            id_field='yandexuid',
            lals_params=[LaLParams(
                name='kinopoisk_online_users',
                id=1377,
                type='lal_internal',
                coverage=5000000,
                include_input=False,
            )],
        )
