#!/usr/bin/env python
# -*- coding: utf-8 -*-
from cached_property import cached_property
import yt.wrapper as yt

from crypta.lib.python import templater
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import (
    LaLParams,
    LalSampleBuilder
)
from crypta.profile.utils import utils


QUERY_TEMPLATE = """
{% set first_table = zipped_tables[0][0] -%}
{% macro from_stmt(table) -%}
    `{{ table["path"] }}` as {{ table["alias"] }}
{%- endmacro -%}

$limit = 100;

$home = ($other) -> {
    RETURN Yson::ConvertToList($other)[0];
};

$lat = ($other) -> {
    RETURN(Yson::LookupDouble($home($other), "latitude"));
};

$lon = ($other) -> {
    RETURN(Yson::LookupDouble($home($other), "longitude"));
};

$dist = ($cur, $prev) -> {
    RETURN Geo::CalculatePointsDifference($lat($cur), $lon($cur), $lat($prev), $lon($prev));
};

$moves = ($distances) -> {
    RETURN ListLength(
        ListFilter($distances, ($x) -> {RETURN $x > $limit;})
    );
};

$raw_data = (
    SELECT
        {{ first_table["alias"] }}.unified_id AS id,
        {{ first_table["alias"] }}.source_unified_id AS id_type
    FROM {{ from_stmt(first_table) }}
        {% for table1, table2 in zipped_tables -%}
    JOIN {{ from_stmt(table2) }}
    ON {{ table1["alias"] }}.unified_id == {{ table2["alias"] }}.unified_id
    {% endfor %}
    WHERE $moves([
    {% for table1, table2 in zipped_tables -%}
    $dist({{ table1["alias"] }}.manual_homes, {{ table2["alias"] }}.manual_homes),
    {% endfor -%}
    ]) == 1
);

$matched_yuids = (
    SELECT matching.target_id AS yandexuid,
    FROM `{{ cryptaid_yandexuid }}` AS matching
    LEFT SEMI JOIN $raw_data AS raw
    USING(id, id_type)

    UNION ALL

    SELECT id AS yandexuid,
    FROM $raw_data
    WHERE id_type == "yandexuid"
);

INSERT INTO `{{ output }}` WITH TRUNCATE
SELECT
    yandexuid,
    "new_settlers" AS segment_name
FROM $matched_yuids
GROUP BY yandexuid;
"""

HOMEWORK_STATE_DIR = "//home/user_identification/homework/v2/prod/state"


def render_query(tables, output):
    aliased_tables = [
        {"alias": "table{}".format(i), "path": table} for i, table in enumerate(tables)
    ]

    zipped_tables = zip(aliased_tables[:-1], aliased_tables[1:])

    return templater.render_template(
        QUERY_TEMPLATE,
        vars={
            "output": output,
            "zipped_tables": zipped_tables,
            "cryptaid_yandexuid": utils.get_matching_table("crypta_id", "yandexuid")
        },
        strict=True,
    )


def get_state_path(date):
    return yt.ypath_join(HOMEWORK_STATE_DIR, date, "homework_unified_id")


class NewSettlers(LalSampleBuilder):
    weeks = 5

    @cached_property
    def states(self):
        return self.yt.list(HOMEWORK_STATE_DIR, sort=True)[-self.weeks:]

    @cached_property
    def last_state_date(self):
        return self.states[-1]

    def requires(self):
        return {
            str(i): ExternalInput(get_state_path(date))
            for i, date in enumerate(self.states)
        }

    def output(self):
        output = super(NewSettlers, self).output()
        output.date = self.last_state_date
        return output

    def build_segment(self, inputs, output_path):
        self.yql.query(
            query_string=render_query(sorted(x.table for x in inputs.values()), output_path),
            transaction=self.transaction,
            udf_resource_dict={'libcrypta_identifier_udf.so': config.CRYPTA_IDENTIFIERS_UDF_RESOURCE},
        )

        self.yt.set_attribute(self.output().table, 'generate_date', self.last_state_date)

        self.prepare_samples_for_lal(
            input_table=output_path,
            id_field='yandexuid',
            lals_params=[LaLParams(
                name='new_settlers',
                id=19376956,
                type='audience_segments',
                coverage=3000000,
                include_input=False,
            )],
        )
