#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.profile.utils.config import config
from crypta.profile.utils.utils import get_matching_table
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import LaLParams, LalSampleBuilder


switched_iphone_to_samsung_query_template = """
$idfa_crypta = (
    SELECT matching.id as idfa, matching.target_id as crypta_id, idfa.date_end as date_end
    FROM `{idfa_eternal}` as idfa
    INNER JOIN `{idfa_crypta_id}` as matching
    USING(id)
);

$gaid_crypta = (
    SELECT matching.id as gaid, matching.target_id as crypta_id, gaid.date_begin as date_begin
    FROM `{gaid_eternal}` as gaid
    INNER JOIN `{gaid_crypta_id}` as matching
    USING(id)
    WHERE gaid.manufacturer == 'Samsung' AND gaid.date_end == '{date}'
);

$parse = DateTime::Parse("%Y-%m-%d");

$crypta_id = (
    SELECT gaid.crypta_id as id,
    FROM $idfa_crypta as idfa
    INNER JOIN $gaid_crypta as gaid
    USING(crypta_id)
    WHERE
        DateTime::ToDays(DateTime::MakeTimestamp($parse(gaid.date_begin)) - DateTime::MakeTimestamp($parse(idfa.date_end))) <= 30 AND
        DateTime::ToDays(DateTime::MakeTimestamp($parse(gaid.date_begin)) - DateTime::MakeTimestamp($parse(idfa.date_end))) > 0
    GROUP BY gaid.crypta_id
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    target_id as yandexuid,
    'iphone_to_samsung' as segment_name
FROM `{crypta_id_yandexuid}` as matching
LEFT SEMI JOIN $crypta_id as crypta_id
USING(id)
"""


class SwitchedIphoneToSamsung(LalSampleBuilder):
    def requires(self):
        return {
            'idfa_eternal': ExternalInput(config.IDFA_ETERNAL),
            'gaid_eternal': ExternalInput(config.GAID_ETERNAL),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            switched_iphone_to_samsung_query_template.format(
                output_table=output_path,
                idfa_eternal=inputs['idfa_eternal'].table,
                gaid_eternal=inputs['gaid_eternal'].table,
                idfa_crypta_id=get_matching_table('idfa', 'crypta_id'),
                gaid_crypta_id=get_matching_table('gaid', 'crypta_id'),
                crypta_id_yandexuid=get_matching_table('crypta_id', 'yandexuid'),
                date=self.date,
            ),
            transaction=self.transaction,
        )

        self.yt.set_attribute(
            output_path,
            'generate_date',
            self.date
        )

        self.prepare_samples_for_lal(
            input_table=output_path,
            id_field='yandexuid',
            lals_params=[LaLParams(
                name='iphone_to_samsung',
                id=16375081,
                type='audience_segments',
                coverage=2000000,
                include_input=False,
            )],
        )
