#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import datetime

import luigi

from crypta.profile.lib import (
    date_helpers,
    luigi_helpers,
)

from crypta.profile.utils.config import config, testing_config, production_config
from crypta.profile.utils.luigi_utils import YtTableAttributeTarget, AttributeExternalInput, BaseYtTask, OldNodesByNameCleaner


sample_query = """
INSERT INTO `{testing_table}` WITH TRUNCATE
SELECT *
FROM `{production_table}`
WHERE yandexuid % 119 == 110
"""


class MakeSample(BaseYtTask):
    date = luigi.Parameter()
    data_source = luigi.Parameter()
    task_group = 'copy_to_testing'

    def requires(self):
        return {
            'parsed_log': AttributeExternalInput(
                table=os.path.join(
                    config.PARSED_LOGS_YT_DIRECTORY.replace(
                        testing_config.CRYPTA_YT_WORKING_DIR,
                        production_config.CRYPTA_YT_WORKING_DIR,
                    ),
                    self.data_source,
                    self.date,
                ),
                attribute_name='closed',
                attribute_value=True,
            ),
            'cleaner': OldNodesByNameCleaner(
                self.date,
                folder=os.path.join(config.PARSED_LOGS_YT_DIRECTORY, self.data_source),
                lifetime=config.NUMBER_OF_DAYS_TO_KEEP_PARSED_LOGS,
            ),
        }

    def output(self):
        return YtTableAttributeTarget(
            os.path.join(
                config.PARSED_LOGS_YT_DIRECTORY.replace(
                    production_config.CRYPTA_YT_WORKING_DIR,
                    testing_config.CRYPTA_YT_WORKING_DIR,
                ),
                self.data_source,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yql.query(
                query_string=sample_query.format(
                    testing_table=self.output().table,
                    production_table=self.input()['parsed_log'].table,
                ),
                transaction=transaction,
            )
            self.yt.set_attribute(self.output().table, 'closed', True)


class MakeLogSample(luigi.WrapperTask):
    date = luigi.Parameter()
    task_group = 'copy_to_testing'

    def requires(self):
        return {
            log_name: MakeSample(self.date, log_name)
            for log_name in ('metrics', 'reqans', 'bar', 'bb', 'mobile_bar', 'pp')
        }


def main():
    luigi_helpers.setup_external_tasks_retry()

    result = luigi.run(
        [
            '--scheduler-url', config.LUIGI_SCHEDULER_URL,
            '--workers', '3',
            '--date', date_helpers.get_yesterday(str(datetime.date.today())),
        ],
        main_task_cls=MakeLogSample,
    )
    if not result:
        exit(1)


if __name__ == '__main__':
    main()
