# -*- coding: utf-8 -*-

import os

import luigi

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.socdem import get_year_from_birth_date, socdem_storage_schema
from crypta.profile.utils.luigi_utils import BaseYtTask, YtDailyRewritableTarget, ExternalInput


MIN_INT32 = -2147483648
MAX_INT32 = 2147483647


def drive_mapper(row):
    if row['uid']:
        row['uid'] = str(abs(row['uid']))
        yield row


class DriveReducer(object):
    def __init__(self, current_date):
        self.current_date = current_date

    def __call__(self, key, rows):
        result_row = {
            'id': key['uid'],
            'id_type': 'puid',
            'update_time': date_helpers.from_utc_date_string_to_timestamp(self.current_date),
            'source': 'drive',
        }

        genders = set()
        birth_dates = set()

        for row in rows:
            if row['gender'] in ('m', 'f'):
                genders.add(row['gender'])

            if row['birth_date_pass'] and row['birth_date_pass'] == row['birth_date_dl'] and \
                    MIN_INT32 <= row['birth_date_pass'] <= MAX_INT32:
                birth_dates.add(
                    date_helpers.to_date_string(
                        date_helpers.from_timestamp_to_datetime(row['birth_date_pass'])
                    )
                )

        if len(genders) == 1:
            result_row['gender'] = list(genders)[0]

        if len(birth_dates) == 1:
            result_row['birth_date'] = list(birth_dates)[0]
            result_row['year_of_birth'] = get_year_from_birth_date(result_row['birth_date'])

        if result_row.get('gender') or result_row.get('birth_date'):
            yield result_row


class DriveSocdem(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def requires(self):
        return ExternalInput(config.DRIVE_DATA)

    def output(self):
        return YtDailyRewritableTarget(
            os.path.join(
                config.SOCDEM_STORAGE_YT_DIR,
                'puid',
                'drive',
            ),
            self.date,
        )

    def run(self):
        with self.yt.Transaction():
            self.yt.create_empty_table(
                self.output().table,
                schema=socdem_storage_schema,
            )

            self.yt.run_map_reduce(
                drive_mapper,
                DriveReducer(self.date),
                self.input().table,
                self.output().table,
                reduce_by='uid',
            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )
