#!/usr/bin/env python
# -*- coding: utf-8 -*-
from os.path import join

import luigi

from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib.socdem_helpers import socdem_config
from crypta.profile.lib.socdem_helpers.inference_utils.thresholds import calculate_thresholds
from crypta.profile.utils import utils
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    ExternalInput,
    YtDailyRewritableTarget,
)


class FindThresholds(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def requires(self):
        return ExternalInput(join(config.YANDEXUID_PROFILES_EXPORT_DIRECTORY, self.date))

    def output(self):
        return YtDailyRewritableTarget(table=config.CLASSIFICATION_THRESHOLDS_TABLE, date=self.date)

    def send_thresholds_to_yt(self, socdem_segment_type, segments_with_thresholds):
        data = []
        for segment, threshold in segments_with_thresholds:
            data.append({
                'fielddate': self.date,
                'socdem': socdem_segment_type,
                'segment': segment,
                'threshold': threshold,
            })
        yt_helpers.write_stats_to_yt(
            yt_client=self.yt,
            table_path=config.DATALENS_SOCDEM_CLASSIFICATION_THRESHOLDS_TABLE,
            data_to_write=data,
            schema={
                'fielddate': 'string',
                'socdem': 'string',
                'segment': 'string',
                'threshold': 'double',
            },
        )

    def run(self):
        thresholds = calculate_thresholds(
            yt_client=self.yt,
            yql_client=self.yql.yql_client,
            profiles_table=self.input().table,
            histograms_directory=config.HISTOGRAMS_DIRECTORY,
            needed_recalls=socdem_config.needed_recalls,
            needed_total_recalls=socdem_config.needed_total_recalls,
        )

        for socdem_segment_type in socdem_config.yet_another_segment_names_by_label_type:
            segments_with_thresholds = zip(
                socdem_config.yet_another_segment_names_by_label_type[socdem_segment_type],
                thresholds[socdem_segment_type],
            )
            self.send_thresholds_to_yt(socdem_segment_type, segments_with_thresholds)
            for segment, threshold in segments_with_thresholds:
                if threshold < 0.01:
                    self.logger.info('Threshold for segment {} is too low, threshold = {}'.format(segment, threshold))

                utils.update_socdem_threshold(
                    socdem=socdem_segment_type,
                    segment=segment,
                    threshold=max(threshold, 0.01),
                )

        self.yt.create_empty_table(
            path=self.output().table,
            schema={
                'gender': 'any',
                'user_age_6s': 'any',
                'income_5_segments': 'any'
            }
        )

        self.yt.write_table(
            table=self.output().table,
            input_stream=[thresholds],
        )

        self.yt.set_attribute(
            self.output().table,
            'generate_date',
            self.date,
        )
