#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

import luigi

from crypta.profile.lib import date_helpers
from crypta.profile.runners.matching.lib.income.merge_job_search import MergeJobSearch
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    ExternalInput,
    YtDailyRewritableTarget
)
from crypta.profile.utils.utils import report_ml_metrics_to_solomon


merge_auto_ru_query_template = '''
$format_date = DateTime::Format("%Y-%m-%d");

$auto_ru = (
    SELECT DISTINCT
        crypta_id_to_puid.target_id AS puid,
        $format_date(MAX(applications.`timestamp`)) AS update_time,
        MAX_BY(
            CAST(applications.current.borrower_person_profile.income.avg_monthly_income AS Double),
            applications.`timestamp`
        ) AS avg_monthly_income,
    FROM RANGE(
        `{auto_ru_credit_applications}`,
        `{first_accounted_date}`,
        `{last_accounted_date}`
    ) AS applications
    INNER JOIN `{matching_auto_id_to_crypta_id}` AS auto_id_to_crypta_id
    ON SUBSTRING(applications.user_id, 5, 8) == auto_id_to_crypta_id.id
    INNER JOIN `{matching_crypta_id_to_puid}` AS crypta_id_to_puid
    ON auto_id_to_crypta_id.target_id == crypta_id_to_puid.id
    WHERE applications.current.borrower_person_profile.income.avg_monthly_income is not Null
        AND applications.current.borrower_person_profile.income.avg_monthly_income > 5000
        AND applications.current.borrower_person_profile.income.avg_monthly_income < 1000000
    GROUP BY crypta_id_to_puid.target_id
);

$train_update_time_lower_bound = (
    SELECT MIN(update_time)
    FROM (
        SELECT puid, update_time
        FROM $auto_ru
        ORDER BY update_time DESC
        LIMIT {auto_ru_train_size_limit}
    )
);

$A_upper_threshold = (
    SELECT A_upper_threshold
    FROM `{thresholds_table}`
);

$B1_upper_threshold = (
    SELECT B1_upper_threshold
    FROM `{thresholds_table}`
);

$B2_upper_threshold = (
    SELECT B2_upper_threshold
    FROM `{thresholds_table}`
);

$C1_upper_threshold = (
    SELECT C1_upper_threshold
    FROM `{thresholds_table}`
);

INSERT INTO `{output_table}`
WITH TRUNCATE
SELECT
    CASE
        WHEN avg_monthly_income < $A_upper_threshold THEN 'A'
        WHEN avg_monthly_income < $B1_upper_threshold THEN 'B1'
        WHEN avg_monthly_income < $B2_upper_threshold THEN 'B2'
        WHEN avg_monthly_income < $C1_upper_threshold THEN 'C1'
        ELSE 'C2'
    END AS income_segment,
    CASE
        WHEN update_time >= $train_update_time_lower_bound THEN 2.
        ELSE 1.
    END AS weight,
    auto_ru.*
FROM $auto_ru AS auto_ru;
'''


class MergeAutoRu(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def requires(self):
        return {
            'thresholds': MergeJobSearch(self.date),
            'auto_ru_credit_applications': ExternalInput(
                os.path.join(
                    config.AUTO_RU_CREDIT_APPLICATIONS_DAILY_FOLDER,
                    date_helpers.get_date_from_past(self.date, days=1)
                )
            )
        }

    def output(self):
        return YtDailyRewritableTarget(
            table=config.MERGED_AUTO_RU_TABLE,
            date=self.date
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yql.query(
                query_string=merge_auto_ru_query_template.format(
                    auto_ru_credit_applications=config.AUTO_RU_CREDIT_APPLICATIONS_DAILY_FOLDER,
                    first_accounted_date=date_helpers.get_date_from_past(self.date, years=config.TIME_TO_ACCOUNT_INCOME_INFO_FOR_VOTING_YEARS),
                    last_accounted_date=date_helpers.get_date_from_past(self.date, days=1),
                    matching_auto_id_to_crypta_id=config.MATCHING_AUTO_ID_TO_CRYPTA_ID,
                    matching_crypta_id_to_puid=config.MATCHING_CRYPTA_ID_TO_PUID,
                    auto_ru_train_size_limit=100000,
                    update_time=self.date,
                    thresholds_table=self.input()['thresholds']['thresholds'].table,
                    output_table=self.output().table,
                ),
                transaction=transaction,
            )

            report_ml_metrics_to_solomon(
                service=config.SOLOMON_SOCDEM_SOURCES_SERVICE,
                metrics_to_send=[{
                    'labels': {
                        'socdem': 'income',
                        'source': self.__class__.__name__,
                        'metric': 'size',
                    },
                    'value': self.yt.row_count(self.output().table),
                }],
            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )
