# -*- coding: utf-8 -*-

import os
import json

import luigi
from yt.wrapper import with_context

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput, BaseYtTask, YtDailyRewritableTarget


@with_context
def apple_google_apps_merge_mapper(row, context):
    row_info = json.loads(row['value'])

    if context.table_index == 0:
        # google_play
        if row_info != "NOT_FOUND" and row['key']:
            categories = []
            for cat in row_info['subcategory_eng']:
                categories.append(cat.lower())
            yield {
                'bundleId': row['key'].lower(),
                'store': 'google_play',
                'name': row_info['name'],
                'raw_categories': categories,
            }
    else:
        # itunes
        if row_info.get('bundleId'):
            yield {
                'bundleId': row_info['bundleId'].lower(),
                'store': 'itunes',
                'name': row_info['trackName'],
                'raw_categories': [row_info['primaryGenreName'].lower()],
            }


def apple_google_apps_merge_reducer(key, rows):
    # There are itunes apps-duplicates so any row can be selected
    yield next(rows)


class GetMobileAppCategories(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def requires(self):
        return {
            'GooglePlayApplicationsLog': ExternalInput(
                os.path.join(
                    config.MOBILE_APPLICATIONS_DIR,
                    'google_play',
                    'latest',
                    'ru'
                ),
            ),
            'ItunesApplicationsLog': ExternalInput(
                os.path.join(
                    config.MOBILE_APPLICATIONS_DIR,
                    'itunes',
                    'latest',
                ),
            )
        }

    def output(self):
        return YtDailyRewritableTarget(
            config.MOBILE_APP_CATEGORIES,
            self.date,
        )

    def run(self):
        self.yt.config['spec_defaults']['pool'] = config.SEGMENTS_POOL

        with self.yt.Transaction():
            self.yt.create_empty_table(
                self.output().table,
                schema={
                    'store': 'string',
                    'bundleId': 'string',
                    'name': 'string',
                    'raw_categories': 'any',
                },
            )

            self.yt.run_map_reduce(
                apple_google_apps_merge_mapper,
                apple_google_apps_merge_reducer,
                [self.input()['GooglePlayApplicationsLog'].table,
                 self.input()['ItunesApplicationsLog'].table],
                self.output().table,
                reduce_by=('store', 'bundleId'),
            )

            self.yt.run_sort(
                self.output().table,
                sort_by='bundleId'
            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )
