# -*- coding: utf-8 -*-
import os
from ast import literal_eval

import luigi
from yt.wrapper import create_table_switch

from crypta.profile.lib import date_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtDailyRewritableTarget,
    ExternalInput,
)
from crypta.profile.utils.socdem import socdem_storage_schema
from crypta.profile.utils.utils import targets_to_tables


class MobileInstallLogSocdemMapper(object):
    def __init__(self, date):
        self.update_time = date_helpers.from_utc_date_string_to_timestamp(date)

    @staticmethod
    def get_gender(value):
        if value in ('f', 'm'):
            return value
        else:
            return None

    @staticmethod
    def process_age(value):
        if value and isinstance(value, basestring) and value.isdigit():
            age = int(value)
            if 0 < age < 100:
                return age

        return None

    def __call__(self, row):
        # myTarget
        if row['PublisherID'] == '136':
            raw_keys = row.get('UrlParameters_Keys')
            raw_values = row.get('UrlParameters_Values')
            params = {}

            if raw_keys and raw_values:
                try:
                    params = dict(zip(literal_eval(raw_keys),
                                      literal_eval(raw_values)))
                except:
                    pass

            gaid = row.get('ADVID')
            idfa = row.get('IFA')
            age = self.process_age(params.get('age'))
            gender = self.get_gender(params.get('gender'))

            if age or gender:
                if idfa:
                    yield create_table_switch(0)
                    yield {
                        'id': idfa,
                        'id_type': 'idfa',
                        'age': age,
                        'gender': gender,
                        'update_time': self.update_time,
                        'source': 'mobile_install_log',
                    }
                elif gaid:
                    yield create_table_switch(1)
                    yield {
                        'id': gaid,
                        'id_type': 'gaid',
                        'age': age,
                        'gender': gender,
                        'update_time': self.update_time,
                        'source': 'mobile_install_log',
                    }


def merge_reducer(key, rows):
    last_record = max(rows, key=lambda record: record['update_time'])
    yield last_record


class MobileInstallLogSocdem(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def __init__(self, date):
        super(MobileInstallLogSocdem, self).__init__(date)
        self.last_processed_date = self.yt.get_attribute(self.output()['GaidMobileInstallLogSocdem'].table, 'generate_date')
        self.input_dates = date_helpers.generate_date_strings(
            date_helpers.get_tomorrow(self.last_processed_date),
            self.date,
        )

    def requires(self):
        log_dirs = [
            config.METRIKA_MOBILE_INSTALL_LOG_FOLDER,
            config.METRIKA_MOBILE_INSTALL_PRIVATE_LOG_FOLDER,
        ]

        required_tasks = {
            'MobileInstallLog': [],
            'IdfaMobileInstallLogSocdem':  ExternalInput(
                os.path.join(
                    config.SOCDEM_STORAGE_YT_DIR,
                    'idfa',
                    'mobile_install_log',
                )
            ),
            'GaidMobileInstallLogSocdem': ExternalInput(
                os.path.join(
                    config.SOCDEM_STORAGE_YT_DIR,
                    'gaid',
                    'mobile_install_log',
                )
            ),
        }

        for input_date in self.input_dates:
            for log_dir in log_dirs:
                required_tasks['MobileInstallLog'].append(ExternalInput(os.path.join(log_dir, input_date)))

        return required_tasks

    def output(self):
        return {
            'IdfaMobileInstallLogSocdem': YtDailyRewritableTarget(
                os.path.join(
                    config.SOCDEM_STORAGE_YT_DIR,
                    'idfa',
                    'mobile_install_log',
                ),
                self.date,
            ),
            'GaidMobileInstallLogSocdem': YtDailyRewritableTarget(
                os.path.join(
                    config.SOCDEM_STORAGE_YT_DIR,
                    'gaid',
                    'mobile_install_log',
                ),
                self.date,
            ),
        }

    def run(self):
        with self.yt.Transaction(),\
                self.yt.TempTable() as idfa_socdem_table,\
                self.yt.TempTable() as gaid_socdem_table:
            for table in (idfa_socdem_table, gaid_socdem_table):
                self.yt.create_empty_table(
                    table,
                    schema=socdem_storage_schema,
                )

            self.yt.run_map(
                MobileInstallLogSocdemMapper(self.date),
                targets_to_tables(self.input()['MobileInstallLog']),
                [idfa_socdem_table, gaid_socdem_table],
            )

            self.yt.run_sort(
                [self.input()['IdfaMobileInstallLogSocdem'].table, idfa_socdem_table],
                self.output()['IdfaMobileInstallLogSocdem'].table,
                sort_by='id',
            )

            self.yt.run_reduce(
                merge_reducer,
                self.output()['IdfaMobileInstallLogSocdem'].table,
                self.output()['IdfaMobileInstallLogSocdem'].table,
                reduce_by='id',
            )

            self.yt.run_sort(
                [self.input()['GaidMobileInstallLogSocdem'].table, gaid_socdem_table],
                self.output()['GaidMobileInstallLogSocdem'].table,
                sort_by='id',
            )

            self.yt.run_reduce(
                merge_reducer,
                self.output()['GaidMobileInstallLogSocdem'].table,
                self.output()['GaidMobileInstallLogSocdem'].table,
                reduce_by='id',
            )

            self.yt.run_sort(
                self.output()['IdfaMobileInstallLogSocdem'].table,
                sort_by='id',
            )

            self.yt.run_sort(
                self.output()['GaidMobileInstallLogSocdem'].table,
                sort_by='id',
            )

            self.yt.set_attribute(
                self.output()['IdfaMobileInstallLogSocdem'].table,
                'generate_date',
                self.date,
            )

            self.yt.set_attribute(
                self.output()['GaidMobileInstallLogSocdem'].table,
                'generate_date',
                self.date,
            )
