# -*- coding: utf-8 -*-

import os
from functools import partial

import luigi
from yt.wrapper import create_table_switch

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.socdem import is_valid_birth_date, get_year_from_birth_date, socdem_storage_schema
from crypta.profile.utils.luigi_utils import BaseYtTask, YtDailyRewritableTarget, ExternalInput
from crypta.profile.tasks.common.merge_socdem_storage import MergeSocdemStorage

social_provider_id_dict = {
    1: 'vk_id',
    2: 'fb_id',
    6: 'ok_id',
}

provider_names = [
    (1, 'vk'),
    (2, 'fb'),
    (4, 'mr'),
    (5, 'gg'),
    (6, 'ok'),
    (12, 'mt'),
]

provider_names_dict = {provider_id: provider_name for provider_id, provider_name in provider_names}
provider_id_table_index_dict = {provider_info[0]: i for i, provider_info in enumerate(provider_names)}

id_types = ('puid', 'email', 'phone_md5', 'vk_id', 'ok_id', 'fb_id')
id_type_table_index_dict = {id_type: i for i, id_type in enumerate(id_types)}

socialdb_data_schema = {
    'id': 'string',
    'id_type': 'string',
    'gender': 'string',
    'provider_id': 'uint64',
    'birth_date': 'string',
}

normalize_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    Identifiers::NormalizeEmail(email) AS email,
    Identifiers::HashMd5Phone(phone) AS phone,
    uid,
    userid,
    gender,
    birthday,
    provider_id
FROM `{input_table}`
"""


def social_auth_mapper(row, current_year):
    provider_id = int(row['provider_id'])
    socdem_record = {
        'provider_id': provider_id,
    }

    if row['gender'] in ('m', 'f'):
        socdem_record['gender'] = row['gender']

    if row['birthday'] and is_valid_birth_date(row['birthday'], current_year):
        socdem_record['birth_date'] = row['birthday']

    if 'gender' in socdem_record or 'birth_date' in socdem_record:
        puid = row['uid']
        processed_email = row['email']
        processed_phone = row['phone']

        record = socdem_record.copy()
        record['id'] = puid
        record['id_type'] = 'puid'
        yield create_table_switch(id_type_table_index_dict[record['id_type']])
        yield record

        if processed_email:
            record = socdem_record.copy()
            record['id'] = processed_email
            record['id_type'] = 'email'
            yield create_table_switch(id_type_table_index_dict[record['id_type']])
            yield record

        if processed_phone:
            record = socdem_record.copy()
            record['id'] = processed_phone
            record['id_type'] = 'phone_md5'
            yield create_table_switch(id_type_table_index_dict[record['id_type']])
            yield record

        if provider_id in social_provider_id_dict:
            record = socdem_record.copy()
            record['id'] = row['userid']
            record['id_type'] = social_provider_id_dict[provider_id]
            yield create_table_switch(id_type_table_index_dict[record['id_type']])
            yield record


def social_auth_reducer(key, rows, current_date, id_type):
    genders = set()
    birth_dates = set()
    provider_ids = set()

    update_time = date_helpers.from_utc_date_string_to_timestamp(current_date)

    for row in rows:
        if 'gender' in row and row['gender']:
            genders.add(row['gender'])

        if 'birth_date' in row and row['birth_date']:
            birth_dates.add(row['birth_date'])

        provider_ids.add(row['provider_id'])

    final_record = {
        'id': key['id'],
        'id_type': key['id_type'],
        'update_time': update_time,
    }

    if len(genders) == 1:
        final_record['gender'] = list(genders)[0]
    if len(birth_dates) == 1:
        final_record['birth_date'] = list(birth_dates)[0]
        final_record['year_of_birth'] = get_year_from_birth_date(final_record['birth_date'])

    if 'gender' in final_record or 'birth_date' in final_record:
        if id_type == 'puid':
            final_record['source'] = 'socialdb'
            yield final_record
        else:
            for provider_id in provider_ids.intersection(provider_names_dict.keys()):
                yield create_table_switch(provider_id_table_index_dict[provider_id])
                source_record = final_record.copy()
                source_record['source'] = 'socialdb_{}'.format(provider_names_dict[provider_id])
                yield source_record


class SocialAuthSocdem(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'import_socdem_data'

    def requires(self):
        return {
            'MergeSocdemStorage': MergeSocdemStorage(date_helpers.get_tomorrow(self.date)),  # prevent overwriting after validation sample creation
            'SocialdbDump': ExternalInput(os.path.join(config.SOCIALDB_DUMP_DIRECTORY, self.date)),
        }

    @staticmethod
    def _get_intermediate_tables():
        return [
            os.path.join(
                config.SOCIALDB_DATA_STORAGE_FOLDER,
                id_type,
            ) for id_type in id_types
        ]

    def output(self):
        return YtDailyRewritableTarget(os.path.join(config.SOCDEM_STORAGE_YT_DIR, 'puid', 'socialdb'), self.date)

    def run(self):
        current_year = int(self.date.split('-')[0])
        intermediate_tables = self._get_intermediate_tables()

        with self.yt.Transaction() as transaction, self.yt.TempTable() as normalized_data_table:
            self.yql.query(
                query_string=normalize_query.format(
                    input_table=self.input()['SocialdbDump'].table,
                    output_table=normalized_data_table,
                ),
                transaction=transaction,
                udf_resource_dict={'libcrypta_identifier_udf.so': config.CRYPTA_IDENTIFIERS_UDF_RESOURCE},
            )

            for table in intermediate_tables:
                self.yt.create_empty_table(
                    table,
                    schema=socialdb_data_schema,
                )

            self.yt.run_map(
                partial(social_auth_mapper, current_year=current_year),
                normalized_data_table,
                intermediate_tables,
            )

            for i, id_type in enumerate(id_types):
                if id_type == 'puid':
                    output_tables = [os.path.join(config.SOCDEM_STORAGE_YT_DIR, 'puid', 'socialdb')]
                else:
                    output_tables = [
                        os.path.join(config.SOCDEM_STORAGE_YT_DIR, id_type, 'socialdb_{}'.format(provider_name))
                        for provider_id, provider_name in provider_names
                    ]

                for table in output_tables:
                    self.yt.create_empty_table(
                        table,
                        schema=socdem_storage_schema,
                    )

                self.yt.run_map_reduce(
                    None,
                    partial(
                        social_auth_reducer,
                        current_date=self.date,
                        id_type=id_type,
                    ),
                    os.path.join(config.SOCIALDB_DATA_STORAGE_FOLDER, id_type),
                    output_tables,
                    reduce_by=['id', 'id_type'],
                )

                for table in output_tables:
                    if self.yt.row_count(table) == 0:
                        self.yt.remove(table)

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )
