"""
Crypta user profile monitoring runner
Usage:
  main.py [--log-file-path=<path>]
  main.py consistency [--log-file-path=<path>]
  main.py (-h | --help)
Options:
  consistency                   Consistency monitorings, which should be run after midnight
  --log-file-path=<path>        Path for logs
"""

import datetime
import os

import luigi
from docopt import docopt

from crypta.profile.lib import (
    date_helpers,
    luigi_helpers,
)

from crypta.profile.utils.config import config

from crypta.profile.tasks.monitoring.profiles.daily_crypta_id_profiles_monitoring import DailyCryptaidProfilesMonitoring
from crypta.profile.tasks.monitoring.profiles.daily_yandexuid_profiles_monitoring import DailyYandexuidProfilesMonitoring
from crypta.profile.tasks.monitoring.profiles.profiles_for_14_days_crypta_id_monitoring import CryptaidProfilesFor14DaysMonitoring
from crypta.profile.tasks.monitoring.profiles.profiles_for_14_days_yandexuid_monitoring import YandexuidProfilesFor14DaysMonitoring
from crypta.profile.tasks.monitoring.profiles.not_in_lab_segments_monitoring import NotInLabSegmentsMonitoring

from crypta.profile.tasks.monitoring.stats.segment_freshness_monitoring import SegmentFreshnessMonitoring

from crypta.profile.tasks.monitoring.consistency.bb_consistency_monitoring import BbConsistencyMonitoring
from crypta.profile.tasks.monitoring.consistency.metrica_consistency_monitoring import MetricaConsistencyMonitoring

from crypta.profile.tasks.monitoring.stats.chevent_log_monitoring import CheventLogMonitoring
from crypta.profile.tasks.monitoring.stats.devid_profiles_socdem_monitoring import DevidProfilesSocdemMonitoring
from crypta.profile.tasks.monitoring.stats.direct_segments_usage_monitoring import DirectSegmentsUsageMonitoring
from crypta.profile.tasks.monitoring.stats.prism_clusters_description_monitoring import PrismClustersDescriptionMonitoring
from crypta.profile.tasks.monitoring.stats.prism_offline_weights_freshness_monitoring import PrismOfflineWeightsFreshnessMonitoring
from crypta.profile.tasks.monitoring.stats.socdem_sample_monitoring import SocdemSampleMonitoring
from crypta.profile.tasks.monitoring.stats.socdem_storage_monitoring import SocdemStorageMonitoring


class MonitoringTasks(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date
        yesterday = date_helpers.get_yesterday(self.date)
        two_days_ago = date_helpers.get_yesterday(yesterday)

        return [
            CryptaidProfilesFor14DaysMonitoring(today),
            CryptaidProfilesFor14DaysMonitoring(yesterday),
            CheventLogMonitoring(yesterday),
            DailyCryptaidProfilesMonitoring(today),
            DailyCryptaidProfilesMonitoring(yesterday),
            DailyYandexuidProfilesMonitoring(today),
            DailyYandexuidProfilesMonitoring(yesterday),
            DevidProfilesSocdemMonitoring(today),
            DirectSegmentsUsageMonitoring(yesterday),
            NotInLabSegmentsMonitoring(today),
            PrismClustersDescriptionMonitoring(two_days_ago),
            PrismOfflineWeightsFreshnessMonitoring(yesterday),
            SegmentFreshnessMonitoring(yesterday, config.REGULAR_SEGMENTS),
            SegmentFreshnessMonitoring(yesterday, config.TRAINABLE_SEGMENTS),
            SocdemSampleMonitoring(yesterday),
            SocdemStorageMonitoring(today),
            YandexuidProfilesFor14DaysMonitoring(today),
            YandexuidProfilesFor14DaysMonitoring(yesterday),
        ]


class ConsistencyMonitoringTasks(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date

        return [
            MetricaConsistencyMonitoring(today),
            BbConsistencyMonitoring(today),
        ]


def main():
    arguments = docopt(__doc__)
    config.DEFAULT_POOL = config.MONITORING_POOL

    logdir = arguments['--log-file-path']
    if logdir:
        config.LOCAL_LOGS_DIRECTORY = logdir
        config.TASKS_LOGS_DIRECTORY = logdir
        config.LOCAL_STORAGE_DIRECTORY = logdir

        if not os.path.exists(logdir):
            os.makedirs(logdir, 0o755)

    luigi_helpers.setup_external_tasks_retry()

    params = [
        '--scheduler-url', config.LUIGI_SCHEDULER_URL,
        '--workers', '4',
        '--date', str(datetime.date.today()),
        '--log-level', 'INFO',
    ]

    if arguments['consistency']:
        luigi.run(
            params,
            main_task_cls=ConsistencyMonitoringTasks,
        )
    else:
        result = luigi.run(
            params,
            main_task_cls=MonitoringTasks,
        )
        if not result:
            exit(1)


if __name__ == '__main__':
    main()
