#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Crypta user profile segments builder

Usage:
  main.py [--log-file-path=<path>] [--task=<task_name>] [--date=<date>]
  main.py coded [--log-file-path=<path>] [--date=<date>]
  main.py standard [--log-file-path=<path>] [--date=<date>]
  main.py constructor [--log-file-path=<path>] [--date=<date>]
  main.py trainable [--log-file-path=<path>] [--date=<date>]
  main.py (-h | --help)

Options:
  --log-file-path=<path>        Path for logs
  --task=<task_name>            Class Task Name
  --date=<date>                 Date string
"""

import datetime
import logging
import os

import luigi
from docopt import docopt
import yt.logger as yt_logger

from crypta.profile.lib import (
    date_helpers,
    luigi_helpers,
)
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config

from crypta.profile.runners.segments.lib.coded_segments.common.active_internet_users import ActiveInternetUsers
from crypta.profile.runners.segments.lib.coded_segments.common.autodrivers import AutoDrivers
from crypta.profile.runners.segments.lib.coded_segments.common.cinema_goers import CinemaGoers
from crypta.profile.runners.segments.lib.coded_segments.common.cinema_goers_often import CinemaGoersOften
from crypta.profile.runners.segments.lib.coded_segments.common.deep_visits import DeepVisits
from crypta.profile.runners.segments.lib.coded_segments.common.edadeal_offline_purchases import OfflinePurchases
from crypta.profile.runners.segments.lib.coded_segments.common.eshoppers import Eshoppers, EshoppersMoreThan5K
from crypta.profile.runners.segments.lib.coded_segments.common.have_birthday_soon import HaveBirthdaySoon
from crypta.profile.runners.segments.lib.coded_segments.common.in_app_payers import InAppPayers
from crypta.profile.runners.segments.lib.coded_segments.common.mobile_devices_owners import MobileDevicesOwners
from crypta.profile.runners.segments.lib.coded_segments.common.organization_visitors import OrganizationVisitors
from crypta.profile.runners.segments.lib.coded_segments.common.users_with_smart_tv import UsersWithSmartTv
from crypta.profile.runners.segments.lib.coded_segments.common.with_children_of_unknown_age import WithChildrenOfUnknownAge

from crypta.profile.runners.segments.lib.coded_segments.internal_common.alice_device_owners import AliceDeviceOwners
from crypta.profile.runners.segments.lib.coded_segments.internal_common.browsers_users import BrowsersUsers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.daily_travellers import DailyTravellers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.eda_decision_makers import EdaDecisionMakers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.metrika_users import MetrikaUsers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.plus_card_owners import PlusCardOwners
from crypta.profile.runners.segments.lib.coded_segments.internal_common.plus_subscribers import PlusSubscribers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.plus_year_subscibers import PlusYearSubscribers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.puids_with_cards import PuidsWithCards
from crypta.profile.runners.segments.lib.coded_segments.internal_common.searched_on_market import SearchedOnMarket
from crypta.profile.runners.segments.lib.coded_segments.internal_common.spravochnik_users import SpravochnikUsers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.uslugi_orgs_owners import UslugiOrganizationsOwners
from crypta.profile.runners.segments.lib.coded_segments.internal_common.weather_pushes_unsubscribers import WeatherPushesUnsubscribers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.yandex_browser_users import YandexBrowserUsers
from crypta.profile.runners.segments.lib.coded_segments.internal_common.yandex_staff import YandexStaff

from crypta.profile.runners.segments.lib.coded_segments.age_segment_18_20 import AgeSegmentEighteenTwenty
from crypta.profile.runners.segments.lib.coded_segments.alice_users import IndeviceAliceUsers, AliceUsers
from crypta.profile.runners.segments.lib.coded_segments.apartment_room_number import ApartmentRoomNumber
from crypta.profile.runners.segments.lib.coded_segments.apps_users import AppsUsers
from crypta.profile.runners.segments.lib.coded_segments.artists import Artists
from crypta.profile.runners.segments.lib.coded_segments.auto_interactions import AutoInteractions
from crypta.profile.runners.segments.lib.coded_segments.avia_travellers import AviaTravellers
from crypta.profile.runners.segments.lib.coded_segments.bank_cards import BankCards, SeveralCardsUsers
from crypta.profile.runners.segments.lib.coded_segments.bought_two_tickets import BoughtTwoTickets
from crypta.profile.runners.segments.lib.coded_segments.business_travellers import BusinessTravellers
from crypta.profile.runners.segments.lib.coded_segments.children_age_segment_clarification import ChildrenAgeSegmentClarification
from crypta.profile.runners.segments.lib.coded_segments.compulsory_auto_insurance import InterestedInCompulsoryAutoInsurance
from crypta.profile.runners.segments.lib.coded_segments.connection_type import ConnectionType
from crypta.profile.runners.segments.lib.coded_segments.console_gamers import ConsoleGamers
from crypta.profile.runners.segments.lib.coded_segments.contest import ContestParticipants
from crypta.profile.runners.segments.lib.coded_segments.custom_classification.custom_classification import CustomClassification
from crypta.profile.runners.segments.lib.coded_segments.custom_regression.custom_regression import CustomRegression
from crypta.profile.runners.segments.lib.coded_segments.devices_without_google_services import WithoutGoogleServices
from crypta.profile.runners.segments.lib.coded_segments.digital_viewers import DigitalViewers
from crypta.profile.runners.segments.lib.coded_segments.direct_clients_by_industry import DirectClientsByIndustry
from crypta.profile.runners.segments.lib.coded_segments.direct_product_users import DirectProductUsers
from crypta.profile.runners.segments.lib.coded_segments.disk_users import DiskUsers
from crypta.profile.runners.segments.lib.coded_segments.ecommerce_owners import EcommerceOwners
from crypta.profile.runners.segments.lib.coded_segments.edadeal_offline_purchases_lal import OfflinePurchasesLookAlike
from crypta.profile.runners.segments.lib.coded_segments.expensive_car_customers import ExpensiveCarCustomers
from crypta.profile.runners.segments.lib.coded_segments.film_lovers_by_genres import FilmLoversByGenres
from crypta.profile.runners.segments.lib.coded_segments.gas_stations import GasStations
from crypta.profile.runners.segments.lib.coded_segments.industry_representatives import IndustryRepresentatives
from crypta.profile.runners.segments.lib.coded_segments.kbt_customers import KbtCustomers
from crypta.profile.runners.segments.lib.coded_segments.kfc_visitors import KfcVisitors
from crypta.profile.runners.segments.lib.coded_segments.kinopoisk_logins import KinopoiskLogins
from crypta.profile.runners.segments.lib.coded_segments.kinopoisk_movie_watchers import KinopoiskMovieWatchers
from crypta.profile.runners.segments.lib.coded_segments.kz_users import KzUsers
from crypta.profile.runners.segments.lib.coded_segments.laptop_users import LaptopUsers
from crypta.profile.runners.segments.lib.coded_segments.logged_in_for_plus import LoggedInForPlus
from crypta.profile.runners.segments.lib.coded_segments.longterm_interest_mobile_gamers import LongtermInterestMobileGamers
from crypta.profile.runners.segments.lib.coded_segments.loyal_to_launcher_install import LoyalToLauncherInstall
from crypta.profile.runners.segments.lib.coded_segments.macos_users import MacOSUsers
from crypta.profile.runners.segments.lib.coded_segments.mail_data import PrepareMailSampleForLalSegments
from crypta.profile.runners.segments.lib.coded_segments.manufacturer_phone_owners import ManufacturerPhoneOwners
from crypta.profile.runners.segments.lib.coded_segments.marketplaces_ltv_users import MarketplacesLtvUsers
from crypta.profile.runners.segments.lib.coded_segments.mobile_gamers import MobileGamers
from crypta.profile.runners.segments.lib.coded_segments.mobile_operators_users import MobileOperatorsUsers
from crypta.profile.runners.segments.lib.coded_segments.mobile_operators_users_by_prefix import MobileOperatorsUsersByPrefix
from crypta.profile.runners.segments.lib.coded_segments.multidevice import Multidevice
from crypta.profile.runners.segments.lib.coded_segments.multidevice_puid import MultidevicePuid
from crypta.profile.runners.segments.lib.coded_segments.music_genres_listeners import MusicGenresListeners
from crypta.profile.runners.segments.lib.coded_segments.nestle_regions import NestleRegions
from crypta.profile.runners.segments.lib.coded_segments.phone_buyers import PhoneBuyers
from crypta.profile.runners.segments.lib.coded_segments.phone_owners import PhoneOwners
from crypta.profile.runners.segments.lib.coded_segments.phone_with_esim_owners import PhoneWithEsimOwners
from crypta.profile.runners.segments.lib.coded_segments.potential_aon_android_users import PotentialAonUsers
from crypta.profile.runners.segments.lib.coded_segments.potential_aon_ios_users import PotentialAonIosUsers
from crypta.profile.runners.segments.lib.coded_segments.preinstalled_apps import PreinstalledApps
from crypta.profile.runners.segments.lib.coded_segments.prism import Prism
from crypta.profile.runners.segments.lib.coded_segments.proleads import Proleads
from crypta.profile.runners.segments.lib.coded_segments.realty_interactions import RealtyInteractions
from crypta.profile.runners.segments.lib.coded_segments.recent_passport_accounts import RecentPassportAccounts
from crypta.profile.runners.segments.lib.coded_segments.score_users_for_telephony import ScoreUsersForTelephony
from crypta.profile.runners.segments.lib.coded_segments.searched_for_phone_numbers import SearchedForPhoneNumbers
from crypta.profile.runners.segments.lib.coded_segments.searched_radisson_on_maps import SearchedRadisson
from crypta.profile.runners.segments.lib.coded_segments.seo_specialists import SeoSpecialists
from crypta.profile.runners.segments.lib.coded_segments.seo_users import SeoUsers
from crypta.profile.runners.segments.lib.coded_segments.smart_gadgets_customers import SmartGadgetsCustomers
from crypta.profile.runners.segments.lib.coded_segments.smokers import Smokers
from crypta.profile.runners.segments.lib.coded_segments.summer_residents import SummerResidents
from crypta.profile.runners.segments.lib.coded_segments.travellers import Travellers
from crypta.profile.runners.segments.lib.coded_segments.video_bloggers import Videobloggers
from crypta.profile.runners.segments.lib.coded_segments.want_to_change_the_provider import WantToChangeTheProvider
from crypta.profile.runners.segments.lib.coded_segments.webmaster import Webmaster
from crypta.profile.runners.segments.lib.coded_segments.widgets import Widgets
from crypta.profile.runners.segments.lib.coded_segments.with_children_by_ages import WithChildrenByAges

from crypta.profile.runners.segments.lib.constructor_segments.build_constructor_segments import BuildConstructorSegments
from crypta.profile.runners.segments.lib.constructor_segments.lal import GetLalSegments


class BuildCodedSegmentsTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date
        yesterday = date_helpers.get_yesterday(today)

        if config.environment == 'testing':
            return [
                DirectProductUsers(today),
                Prism(date_helpers.get_date_from_past(today, days=2)),
            ]
        else:
            return [
                ActiveInternetUsers(yesterday),
                AgeSegmentEighteenTwenty(today),
                AliceDeviceOwners(yesterday),
                AliceUsers(yesterday),
                ApartmentRoomNumber(yesterday),
                AppsUsers(yesterday),
                Artists(yesterday),
                AutoDrivers(yesterday),
                AutoInteractions(yesterday),
                AviaTravellers(yesterday),
                BankCards(yesterday),
                BoughtTwoTickets(yesterday),
                BrowsersUsers(today),
                BusinessTravellers(yesterday),
                ChildrenAgeSegmentClarification(today),
                CinemaGoers(yesterday),
                CinemaGoersOften(yesterday),
                ConnectionType(yesterday),
                ConsoleGamers(yesterday),
                ContestParticipants(yesterday),
                DailyTravellers(yesterday),
                DeepVisits(yesterday),
                DigitalViewers(yesterday),
                DirectClientsByIndustry(yesterday),
                DirectProductUsers(today),
                DiskUsers(yesterday),
                EcommerceOwners(today),
                EdaDecisionMakers(yesterday),
                Eshoppers(yesterday),
                EshoppersMoreThan5K(yesterday),
                ExpensiveCarCustomers(yesterday),
                FilmLoversByGenres(yesterday),
                GasStations(yesterday),
                HaveBirthdaySoon(yesterday),
                UsersWithSmartTv(yesterday),
                InAppPayers(yesterday),
                IndeviceAliceUsers(yesterday),
                IndustryRepresentatives(yesterday),
                InterestedInCompulsoryAutoInsurance(yesterday),
                KbtCustomers(yesterday),
                KfcVisitors(yesterday),
                KinopoiskLogins(yesterday),
                KinopoiskMovieWatchers(yesterday),
                KzUsers(today),
                LaptopUsers(yesterday),
                LoggedInForPlus(yesterday),
                LongtermInterestMobileGamers(yesterday),
                LoyalToLauncherInstall(yesterday),
                MacOSUsers(yesterday),
                ManufacturerPhoneOwners(today),
                ManufacturerPhoneOwners(today),
                MarketplacesLtvUsers(yesterday),
                MetrikaUsers(yesterday),
                MobileDevicesOwners(yesterday),
                MobileGamers(yesterday),
                MobileOperatorsUsers(yesterday),
                MobileOperatorsUsersByPrefix(yesterday),
                Multidevice(yesterday),
                MultidevicePuid(yesterday),
                MusicGenresListeners(yesterday),
                NestleRegions(yesterday),
                OfflinePurchases(yesterday),
                OfflinePurchasesLookAlike(today),
                OrganizationVisitors(yesterday),
                PhoneBuyers(yesterday),
                PhoneOwners(yesterday),
                PhoneWithEsimOwners(today),
                PlusCardOwners(today),
                PlusSubscribers(yesterday),
                PlusYearSubscribers(yesterday),
                PotentialAonIosUsers(yesterday),
                PotentialAonUsers(yesterday),
                PreinstalledApps(yesterday),
                PrepareMailSampleForLalSegments(today),
                Prism(date_helpers.get_date_from_past(today, days=2)),
                Proleads(yesterday),
                PuidsWithCards(yesterday),
                RealtyInteractions(yesterday),
                RecentPassportAccounts(yesterday),
                ScoreUsersForTelephony(today),
                SearchedForPhoneNumbers(yesterday),
                SearchedOnMarket(yesterday),
                SearchedRadisson(yesterday),
                SeoSpecialists(yesterday),
                SeoUsers(yesterday),
                SeveralCardsUsers(yesterday),
                SmartGadgetsCustomers(yesterday),
                Smokers(yesterday),
                SpravochnikUsers(yesterday),
                SummerResidents(yesterday),
                Travellers(yesterday),
                UslugiOrganizationsOwners(today),
                Videobloggers(yesterday),
                WantToChangeTheProvider(yesterday),
                WeatherPushesUnsubscribers(yesterday),
                Webmaster(yesterday),
                Widgets(yesterday),
                WithChildrenByAges(yesterday),
                WithChildrenOfUnknownAge(yesterday),
                WithoutGoogleServices(today),
                YandexBrowserUsers(yesterday),
                YandexStaff(yesterday),
            ]


class BuildConstructorSegmentsWithLalsTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        yesterday = date_helpers.get_yesterday(self.date)

        return [
            BuildConstructorSegments(date=yesterday),
            GetLalSegments(date=yesterday),
        ]


class TrainableSegmentsTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date

        return [
            CustomClassification(today),
            CustomRegression(today),
        ]


def main():
    arguments = docopt(__doc__)

    luigi_utils.configure_luigi_stdout_logger()
    luigi_helpers.setup_external_tasks_retry()

    config.DEFAULT_POOL = config.SEGMENTS_POOL

    if arguments['--log-file-path']:
        config.LOCAL_LOGS_DIRECTORY = config.TASKS_LOGS_DIRECTORY = arguments['--log-file-path']
        config.LOCAL_STORAGE_DIRECTORY = arguments['--log-file-path']

    date = str(datetime.date.today())
    if arguments['--date']:
        date = arguments['--date']

    # TODO(r-andrey): when new schedulers for 'coded' and 'constructor' replace old ones, remove 'standard' and empty.
    # TODO(r-andrey): remove args from __docopt__ as well; leave the support for manual run via '--task', though.
    if arguments['coded']:
        task = BuildCodedSegmentsTask.__name__
    elif arguments['standard'] or arguments['constructor']:
        task = BuildConstructorSegmentsWithLalsTask.__name__
    elif arguments['trainable']:
        config.DEFAULT_POOL = config.TRAINABLE_SEGMENTS_POOL
        task = TrainableSegmentsTask.__name__
    elif arguments['--task']:
        task = arguments['--task']
    else:
        task = BuildCodedSegmentsTask.__name__

    number_of_workers_by_task_name = {
        BuildConstructorSegmentsWithLalsTask.__name__: 5,
        TrainableSegmentsTask.__name__: 4,
        BuildCodedSegmentsTask.__name__: 5,
    }

    yt_logger.LOGGER.setLevel(logging.DEBUG)
    yt_logger.LOGGER.handlers = [logging.FileHandler(os.path.join(config.LOCAL_LOGS_DIRECTORY, "yt.log"))]

    luigi.run(
        [
            task,
            '--scheduler-url', config.LUIGI_SCHEDULER_URL,
            '--workers', str(number_of_workers_by_task_name.get(task, 1)),
            '--date', date,
            '--log-level', 'INFO',
        ],
    )


if __name__ == '__main__':
    main()
