#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


alice_sessions_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    SUBSTRING(`uuid`, 3) AS id,
    'uuid' AS id_type,
    AGGREGATE_LIST_DISTINCT(app)[0] AS apps,
    COUNT(*) AS n_sessions,
    SOME(fielddate) AS `date`
FROM `{input_table}`
GROUP BY `uuid`
"""


class ProcessedDialogSessionsFromAliceUsers(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.ALICE_DIALOG_LOGS,
                self.date,
            ),
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            alice_sessions_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


indevice_alice_users_segment_query = """
$all_data = (
    SELECT
        id,
        id_type,
        SUM(n_sessions) AS n_sessions,
        AGGREGATE_LIST_DISTINCT(`date`) AS dates,
        AGGREGATE_LIST_DISTINCT(apps)[0] AS app
    FROM `{input_table}`
    GROUP BY id, id_type
);

$alice_users = (
    SELECT
        id,
        id_type,
        'alice_users' AS segment_name
    FROM $all_data
    WHERE ListLength(dates) >= 2 OR n_sessions >= 20
);

$alice_users_with_screens = (
    SELECT
        id,
        id_type,
        'alice_users_with_screens' AS segment_name
    FROM $all_data
    WHERE (ListLength(dates) >= 2 OR n_sessions >= 20)
        AND app IN ('search_app_prod', 'browser_prod', 'yabro_prod', 'search_app_beta',
                    'yabro_beta', 'browser_alpha', 'browser_beta')
);

$active_alice_users_raw = (
    SELECT
        id,
        id_type,
        app
    FROM $all_data
    WHERE ListLength(dates) >= 5
);

$active_alice_users_crypta_id = (
    SELECT
        cryptaId,
        AGGREGATE_LIST_DISTINCT(app) AS apps
    FROM (
        SELECT
            users.app AS app,
            matching.cryptaId AS cryptaId
        FROM $active_alice_users_raw AS users
        INNER JOIN `{id_to_crypta_id_table}` AS matching
        USING (id, id_type)
    )
    GROUP BY cryptaId
);

$active_alice_users_without_navi_and_auto = (
    SELECT cryptaId
    FROM $active_alice_users_crypta_id
    WHERE ListLength(ListFilter(apps, ($name) -> {{RETURN $name != 'auto' AND $name != 'navigator';}})) > 0
);

$active_alice_users = (
    SELECT
        matching.id AS id,
        matching.id_type AS id_type,
        'active_alice_users' AS segment_name
    FROM $active_alice_users_without_navi_and_auto AS users
    INNER JOIN `{crypta_id_to_id_table}` AS matching
    USING (cryptaId)
    WHERE matching.id_type == 'yandexuid'
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT *
FROM $active_alice_users
UNION ALL
SELECT *
FROM $alice_users
UNION ALL
SELECT *
FROM $alice_users_with_screens;
"""


alice_users_segment_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    'alice_users' AS segment_name
FROM (
    SELECT
        id,
        id_type,
        SUM(n_sessions) AS n_sessions,
        AGGREGATE_LIST_DISTINCT(`date`) AS dates
    FROM `{input_table}`
    GROUP BY id, id_type
)
WHERE ListLength(dates) >= 2 OR n_sessions >= 20
"""


class IndeviceAliceUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'alice_users': 7059511,
        'active_alice_users': 9054388,
        'alice_users_with_screens': 11207263,
    }
    indevice = True
    keyword = 557
    number_of_days = 35

    def requires(self):
        return {
            'ProcessedDialogs': LogProcessor(
                ProcessedDialogSessionsFromAliceUsers,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            indevice_alice_users_segment_query.format(
                input_table=inputs['ProcessedDialogs'].table,
                id_to_crypta_id_table=config.VERTICES_NO_MULTI_PROFILE,
                crypta_id_to_id_table=config.VERTICES_NO_MULTI_PROFILE_BY_CRYPTA_ID,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class AliceUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'alice_users': 1231,
    }
    keyword = 549
    number_of_days = 35

    def requires(self):
        return {
            'ProcessedDialogs': LogProcessor(
                ProcessedDialogSessionsFromAliceUsers,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            alice_users_segment_query.format(
                input_table=inputs['ProcessedDialogs'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
