import os

from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils import (
    processors,
    builders,
)

day_processor_query = """
$hosts = AsSet(
{hosts}
);

$re = Re2::Capture("rooms?(?:_count)?([0-9])?=([-0-9,]+)");

$get_room_number = ($url) -> {{
    RETURN
        CASE
            WHEN ListLength(String::SplitToList($re(Url::Parse($url).Query)._1, ",")) > 0 THEN String::SplitToList($re(Url::Parse($url).Query)._1, ",")
            WHEN ListLength(String::SplitToList($re(Url::Parse($url).Query)._2, ",")) > 0 THEN String::SplitToList($re(Url::Parse($url).Query)._2, ",")
            ELSE NULL
        END
}};

$rooms = (
    SELECT
        yandexuid,
        $get_room_number(url) AS rooms
    FROM `{input_table}`
    WHERE yandexuid IS NOT NULL AND yandexuid != 0 AND host in $hosts
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT DISTINCT
    yandexuid AS id,
    "yandexuid" AS id_type,
    rooms AS segment_name
FROM (
    SELECT
        *
    FROM $rooms
    WHERE rooms IS NOT NULL
)
FLATTEN LIST BY rooms;
"""


class ProcessedBarForApartmentRoomNumber(processors.DayProcessor):
    def requires(self):
        return luigi_utils.AttributeExternalInput(
            os.path.join(
                config.BAR_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        hosts = [
            'samolet.ru',
            'a101.ru',
            'donstroy.com',
            'pik.ru',
            'cian.ru',
            'fsk.ru',
        ]

        self.yql.query(
            day_processor_query.format(
                hosts='\n'.join("'{}',".format(host) for host in hosts),
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = u"""
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT DISTINCT
    CAST(id AS String) AS id,
    id_type,
    segment_name
FROM `{input_table}`
WHERE segment_name IN AsSet("-1", "9", "1", "2", "3", "4");
"""


class ApartmentRoomNumber(builders.RegularSegmentBuilder):
    name_segment_dict = {
        '-1': (557, 18815938),
        '9': (557, 18815938),
        '1': (557, 18824788),
        '2': (557, 18824791),
        '3': (557, 18824800),
        '4': (557, 18824809),
    }
    number_of_days = 35

    def requires(self):
        return processors.LogProcessor(
            ProcessedBarForApartmentRoomNumber,
            self.date,
            self.number_of_days,
        )

    def build_segment(self, inputs, output_path):
        self.yql.query(
            segment_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
