#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
from functools import partial

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


day_processor_query = u"""
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    `APIKey` AS api_key,
    `UUID` AS id,
    MAX(StartDate) AS `date`
FROM CONCAT(`{superapp_table}`, `{browser_table}`, `{yandex_events_table}`)
WHERE APIKey IN ({api_keys}) AND SessionType IS NULL
GROUP BY `APIKey`, `UUID`;
"""

API_KEYS = {
    '10321': 'search',  # ru.yandex.searchplugin
    '42989': 'search_ios',  # ru.yandex.mobile
    '106400': 'bro',  # com.yandex.browser
}


class MetrikaMobileLogForAppsUsers(DayProcessor):
    def requires(self):
        return {
            'yandex_events': ExternalInput(
                os.path.join(
                    config.APPMETRICA_YANDEX_EVENTS_LOG_FOLDER,
                    self.date,
                ),
            ),
            'browser': ExternalInput(
                os.path.join(
                    config.BROWSER_METRIKA_MOBILE_LOG_FOLDER,
                    self.date,
                ),
            ),
            'superapp': ExternalInput(
                os.path.join(
                    config.SUPERAPP_METRIKA_MOBILE_LOG_FOLDER,
                    self.date,
                ),
            ),
        }

    def process_day(self, inputs, output_path):
        self.yql.query(
            query_string=day_processor_query.format(
                superapp_table=inputs['superapp'].table,
                browser_table=inputs['browser'].table,
                yandex_events_table=inputs['yandex_events'].table,
                output_table=output_path,
                api_keys=", ".join(API_KEYS),
            ),
            transaction=self.transaction,
        )


def reducer(key, rows, current_date):
    dates = set(row['date'] for row in rows)
    segment_name_prefix = API_KEYS[str(key['api_key'])]

    last_date = date_helpers.from_timestamp_to_datetime(max(dates))
    days_passed = (current_date - last_date).days

    if segment_name_prefix in ('search', 'bro'):
        segment_name_suffix = None

        if days_passed < 16:
            segment_name_suffix = '0_15'
        elif days_passed < 31:
            segment_name_suffix = '16_30'
        elif days_passed < 61:
            segment_name_suffix = '31_60'

        if segment_name_suffix is not None:
            yield {
                'id': key['id'],
                'id_type': 'uuid',
                'segment_name': segment_name_prefix + '__' + segment_name_suffix,
            }

    if segment_name_prefix == 'search':
        segment_name_suffix = None

        if days_passed < 7:
            segment_name_suffix = '0_6'
        elif days_passed < 16:
            segment_name_suffix = '7_15'

        if segment_name_suffix is not None:
            yield {
                'id': key['id'],
                'id_type': 'uuid',
                'segment_name': segment_name_prefix + '__' + segment_name_suffix,
            }

    if segment_name_prefix.startswith('search'):
        if len(dates) >= 15:
            yield {
                'id': key['id'],
                'id_type': 'uuid',
                'segment_name': 'search_active_users',
            }


class AppsUsers(RegularSegmentBuilder):
    name_segment_dict = {
        # APIkey
        'search__0_6': 1650,
        'search__7_15': 1651,
        'search__0_15': 1271,
        'search__16_30': 1272,
        'search__31_60': 1273,
        'bro__0_15': 1268,
        'bro__16_30': 1269,
        'bro__31_60': 1270,
        'search_active_users': 1770,
    }

    keyword = 549
    number_of_days = 60

    def requires(self):
        return LogProcessor(
            MetrikaMobileLogForAppsUsers,
            self.date,
            self.number_of_days,
        )

    def build_segment(self, inputs, output_path):
        self.yt.run_map_reduce(
            None,
            partial(
                reducer,
                current_date=date_helpers.from_date_string_to_datetime(self.date, date_helpers.DATE_FORMAT),
            ),
            inputs.table,
            output_path,
            reduce_by=('api_key', 'id')
        )
