#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInputDate
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder

from crypta.profile.runners.segments.lib.coded_segments.data.bank_card_types import classes


class BankCardsMapper(object):
    def __init__(self, classes):
        self.classes = classes

    @staticmethod
    def make_row(row, segment):
        return {
            'id': str(row['uid']),
            'id_type': 'puid',
            'segment_name': segment,
        }

    def __call__(self, row):
        system = row['system']
        is_expired = row['is_expired']
        if system in ('MasterCard', 'VISA') and is_expired != 1:
            if system == 'MasterCard':
                yield self.make_row(row, 'mastercard')
            else:
                yield self.make_row(row, 'visa')

            card_bin = int(row['BIN'])
            if card_bin in self.classes:
                yield self.make_row(row, self.classes[card_bin])

        elif system == 'MIR' and is_expired != 1:
            yield self.make_row(row, 'mir')


class BankCards(RegularSegmentBuilder):
    name_segment_dict = {
        'BE': (557, 13048747),
        'BusCor': (557, 13048750),
        'BusPr': (557, 13048753),
        'BusST': (557, 13048756),
        'Go': (557, 13048759),
        'Pl': (557, 13048765),
        'St': (557, 13048768),
        'WE': (557, 13048771),
        'Wo': (557, 13048774),
        'Mae': (557, 15698368),
        'mastercard': (547, 1971),
        'visa': (547, 1972),
        'mir': (557, 19190386),
        'visa_gold': (557, 19619093),
        'visa_platinum': (557, 19619094),
        'visa_signature': (557, 19619095),
        'visa_infinite': (557, 19619096),
    }

    def requires(self):
        return ExternalInputDate(config.PUIDS_WITH_CARDS, self.date)

    def build_segment(self, inputs, output_path):
        self.yt.run_map(
            BankCardsMapper(classes),
            self.input().table,
            output_path,
        )

        self.yt.set_attribute(
            output_path,
            'generate_date',
            self.date,
        )


def several_cards_reducer(key, rows):
    bins = set()

    for row in rows:
        if not row['is_expired']:
            bins.add(row['BIN'])

    if len(bins) == 1:
        yield {
            'id': str(key['uid']),
            'id_type': 'puid',
            'segment_name': 'one_card'
        }

    elif len(bins) > 1:
        yield {
            'id': str(key['uid']),
            'id_type': 'puid',
            'segment_name': 'several_cards'
        }


class SeveralCardsUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'one_card': 17512228,
        'several_cards': 17512237,
    }

    keyword = 557

    def requires(self):
        return ExternalInputDate(config.PUIDS_WITH_CARDS, self.date)

    def build_segment(self, inputs, output_path):
        self.yt.run_map_reduce(
            None,
            several_cards_reducer,
            inputs.table,
            output_path,
            reduce_by='uid'
        )
