#!/usr/bin/env python
# -*- coding: utf-8 -*-

from os.path import join
from collections import defaultdict

from crypta.profile.lib import date_helpers

from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.config import config
from crypta.profile.runners.segments.lib.coded_segments.travellers import Travellers

only_travellers_query = """
$million = AsSet(1, 2, 10174, 213, 65, 54, 47, 43, 56, 66, 51, 39, 62, 50, 193, 38, 35);
$europe = AsSet(21610, 21359, 21203, 20790, 20574, 10089, 10088, 10083, 10077, 10074, 10070,
                  10069, 10068, 10067, 10064, 10063, 10057, 10054, 983, 734, 733, 246, 206,
                  205, 204, 203, 180, 179, 127, 126, 125, 124, 123, 122, 121, 120, 119,
                  118, 117, 116, 115, 114, 113, 102, 96);

$visited_crypta_id = (
    SELECT
        matching.cryptaId AS crypta_id,
        region,
        days
    FROM RANGE(`{storage}`) AS travellers
    INNER JOIN `{id_to_crypta_id_table}` AS matching
    USING(id, id_type)
);

INSERT INTO `{travellers}` WITH TRUNCATE
SELECT
    days,
    crypta_id,
    city,
    country
FROM (
    SELECT
        storage.days AS days,
        storage.crypta_id AS crypta_id,
        region,
        Geo::RoundRegionById(storage.region, 'city').id AS city,
        Geo::RoundRegionById(storage.region, 'country').id AS country,
        UNWRAP(CAST(info.main_region_obl as Int32)) AS main_region_obl
    FROM $visited_crypta_id AS storage
    INNER JOIN `{crypta_id_regions}` AS info
    USING(crypta_id)
    WHERE info.main_region_obl IS NOT NULL
)
WHERE
    NOT Geo::IsRegionInRegion(region, main_region_obl) AND
    (city IN $million OR country IN $europe)
ORDER BY crypta_id;
"""


class BusinessTravellersReducer(object):
    def __init__(self, date):
        self.tstamp = date_helpers.from_date_string_to_timestamp(date)

    def __call__(self, key, rows):

        million = {1, 2, 10174, 213, 65, 54, 47, 43, 56, 66, 51, 39, 62, 50, 193, 38, 35}
        europe = {21610, 21359, 21203, 20790, 20574, 10089, 10088, 10083, 10077, 10074, 10070,
                  10069, 10068, 10067, 10064, 10063, 10057, 10054, 983, 734, 733, 246, 206,
                  205, 204, 203, 180, 179, 127, 126, 125, 124, 123, 122, 121, 120, 119,
                  118, 117, 116, 115, 114, 113, 102, 96}

        quater_duration = 91 * 24 * 60 * 60  # для деления года на кварталы

        region_history = defaultdict(list)
        country_history = defaultdict(list)

        for row in rows:
            trip_start = date_helpers.from_date_string_to_timestamp(sorted(row['days'])[0])

            if row['city'] in million:
                history = region_history
                region = row['city']
            elif row['country'] in europe:
                history = country_history
                region = row['country']
            else:
                continue

            if trip_start >= self.tstamp - quater_duration:
                history[0].append(region)
            elif self.tstamp - quater_duration >= trip_start >= self.tstamp - quater_duration * 2:
                history[1].append(region)
            elif self.tstamp - quater_duration * 2 >= trip_start >= self.tstamp - quater_duration * 3:
                history[2].append(region)
            elif self.tstamp - quater_duration * 3 >= trip_start >= self.tstamp - quater_duration * 4:
                history[3].append(region)

        if (0 in region_history or 0 in country_history) and \
                (1 in region_history or 1 in country_history) and \
                (2 in region_history or 2 in country_history) and \
                (3 in region_history or 3 in country_history):
            yield {
                'id': str(key['crypta_id']),
                'id_type': 'crypta_id',
                'segment_name': 'business_travellers',
            }


class BusinessTravellers(RegularSegmentBuilder):
    name_segment_dict = {
        'business_travellers': 12676609,
    }
    keyword = 557

    def requires(self):
        return Travellers(self.date)

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as travellers:
            self.yql.query(
                only_travellers_query.format(
                    storage=join(config.DAY_PROCESSORS_OUTPUT_FOLDER, 'TravellersWeekly'),
                    crypta_id_regions=config.GEO_CRYPTA_ID,
                    id_to_crypta_id_table=config.VERTICES_NO_MULTI_PROFILE,
                    travellers=travellers,
                ),
                transaction=self.transaction,
            )

            self.yt.run_reduce(
                BusinessTravellersReducer(self.date),
                travellers,
                output_path,
                reduce_by='crypta_id'
            )
