#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput, ExternalInputDate
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


clarify_children_yql_template = """
$possible_children_by_voice = (
    SELECT `uuid`, TableName() AS `date`, '0_12' AS segment_name
    FROM RANGE(`{biometry_folder}`, `{biometry_first_date}`, `{biometry_last_date}`)
    WHERE bio_child > 0.8
);

$possible_children_by_voice = (
    SELECT DISTINCT `uuid`, `date`, segment_name
    FROM $possible_children_by_voice
);

$possible_children_by_voice = (
    SELECT `uuid`, segment_name
    FROM $possible_children_by_voice
    GROUP BY `uuid`, segment_name
    HAVING COUNT(*) >= 2
);

$sources_new_age = (
    SELECT matching.cryptaId AS cryptaId,
        CASE
            WHEN socdem_storage.birth_date > '{thirteenth_birthday}' THEN '0_12'
            WHEN '{thirteenth_birthday}' >= socdem_storage.birth_date AND
                socdem_storage.birth_date > '{eighteenth_birthday}' THEN '13_17'
            ELSE '18_99'
        END AS segment_name
    FROM `{socdem_storage_table}` AS socdem_storage
    INNER JOIN `{id_to_crypta_id_table}` AS matching
    ON socdem_storage.id == matching.id AND socdem_storage.id_type == matching.id_type
    WHERE socdem_storage.birth_date is not Null
    UNION ALL
    SELECT matching.cryptaId AS cryptaId, biometry.segment_name AS segment_name
    FROM $possible_children_by_voice AS biometry
    INNER JOIN `{id_to_crypta_id_table}` AS matching
    ON biometry.`uuid` == matching.id
    WHERE matching.id_type == 'uuid'
);

$sources_mode_age = (
    SELECT cryptaId, MODE(segment_name)[0].Value AS segment_name
    FROM $sources_new_age
    GROUP BY cryptaId
);

INSERT INTO `{output_table}` WITH TRUNCATE

SELECT matching.id AS id,
    matching.id_type AS id_type,
    sources.segment_name AS segment_name
FROM $sources_mode_age AS sources
INNER JOIN `{crypta_id_to_id_table}` AS matching
USING(cryptaId)
WHERE sources.segment_name in ('0_12', '13_17') AND matching.id_type == 'yandexuid';
"""


class ChildrenAgeSegmentClarification(RegularSegmentBuilder):
    name_segment_dict = {
        '0_12': 1197,
        '13_17': 1196,
    }

    keyword = 547

    def requires(self):
        return {
            'SocdemStorage': ExternalInputDate(config.SOCDEM_STORAGE_TABLE, date=self.date),
            'VoiceSocdemSource': ExternalInput(os.path.join(config.VOICE_SOCDEM_SOURCE, date_helpers.get_yesterday(self.date))),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            clarify_children_yql_template.format(
                biometry_folder=os.path.dirname(inputs['VoiceSocdemSource'].table),
                biometry_first_date=date_helpers.get_date_from_past(self.date, years=1),
                biometry_last_date=self.date,
                socdem_storage_table=inputs['SocdemStorage'].table,
                thirteenth_birthday=date_helpers.get_date_from_past(self.date, years=13),
                eighteenth_birthday=date_helpers.get_date_from_past(self.date, years=18),
                id_to_crypta_id_table=config.VERTICES_NO_MULTI_PROFILE,
                crypta_id_to_id_table=config.VERTICES_NO_MULTI_PROFILE_BY_CRYPTA_ID,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
