#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


MIN_EVENTS_PER_DEVID_FOR_CELL_USERS = 30
MIN_EVENTS_PER_DEVID_FOR_WIFI_USERS = 5
CELL_PERCENTAGE_THRESHOLD = 0.7
WIFI_PERCENTAGE_THRESHOLD = 0.7


class ActiveInternetUsersMapper(object):
    @staticmethod
    def parse_connection_hist(value):
        connection_hist = dict(pair.split(":") for pair in value.split(","))

        return {
            'cell': float(connection_hist.get('cell', 0)),
            'wifi': float(connection_hist.get('wifi', 0)),
        }

    @staticmethod
    def get_segment_name(hist):
        n_events = sum(hist.values())

        if n_events > MIN_EVENTS_PER_DEVID_FOR_CELL_USERS:
            cell_share = round(hist['cell'] / n_events, 3)
            if cell_share > CELL_PERCENTAGE_THRESHOLD:
                return 'active_mobile_users'
        elif n_events > MIN_EVENTS_PER_DEVID_FOR_WIFI_USERS:
            wifi_share = round(hist['wifi'] / n_events, 3)
            if wifi_share > WIFI_PERCENTAGE_THRESHOLD:
                return 'mobile_wifi_users'
        else:
            return None

    def __call__(self, row):
        if row['id'] and row['id_type'] in ('gaid', 'idfa') and row['connection_hist']:
            hist = self.parse_connection_hist(row['connection_hist'])
            segment_name = self.get_segment_name(hist)

            if segment_name is not None:
                yield {
                    'id': row['id'],
                    'id_type': row['id_type'],
                    'segment_name': segment_name,
                }


class ActiveInternetUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'mobile_wifi_users': (547, 1045),
        'active_mobile_users': (547, 1043),
    }

    def requires(self):
        return ExternalInput(
            config.APP_METRICA_MONTH,
            columns=('id', 'id_type', 'connection_hist'),
        )

    def build_segment(self, inputs, output_path):
        self.yt.run_map(
            ActiveInternetUsersMapper(),
            inputs.table,
            output_path,
        )
