#!/usr/bin/env python
# -*- coding: utf-8 -*-
from functools import partial

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


def ticket_mapper(row, today_date):
    if row['event_type'] == 'movie' and row['action_type'] == 'ticket.purchase.approved':
        if date_helpers.get_date_from_future(row['created_dt'], 180) >= today_date:
            if row['yuid']:
                yield {
                    'id': row['yuid'],
                    'id_type': 'yandexuid',
                    'segment_name': 'cinema_goers',
                }
            elif row['puid']:
                yield {
                    'id': str(row['puid']),
                    'id_type': 'puid',
                    'segment_name': 'cinema_goers',
                }


class CinemaGoers(RegularSegmentBuilder):
    name_segment_dict = {
        'cinema_goers': (547, 1977),
    }

    def requires(self):
        return {
            'AfishaTicketOrderLog': ExternalInput(
                config.AFISHA_TICKET_ORDER_LOG,
                columns=('yuid', 'puid', 'event_type', 'action_type', 'created_dt'),
            )
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as ticket_ids:
            self.yt.run_map(
                partial(ticket_mapper, today_date=self.date),
                inputs['AfishaTicketOrderLog'].table,
                ticket_ids,
            )

            self.yt.unique(
                [ticket_ids],
                output_path,
                unique_by=['id', 'id_type', 'segment_name'],
            )
