#!/usr/bin/env python
# -*- coding: utf-8 -*-
from functools import partial
import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor

from crypta.profile.runners.segments.lib.coded_segments.common.organization_visitors import OrganizationCategoryDictionary

day_processor_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    mmetric_devid AS id,
    'mm_device_id' AS id_type,
    permalink,
    TableName() AS `date`
FROM `{input_table}`
WHERE ListLength(
    SetIntersection(
        ToSet(AsList('place.make-route', 'route.make-route', 'DestinationSuggests')),
        ToSet(Yson::ConvertToStringList(methods)),
    )
) > 0
"""


class CinemaVisitsDayProcessor(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.ORG_VISITS_DIRECTORY, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


def ticket_mapper(row, days, today_date):
    if row['event_type'] == 'movie' and row['action_type'] == 'ticket.purchase.approved':
        if date_helpers.get_date_from_future(row['created_dt'], days) >= today_date:
            if row['yuid']:
                yield {
                    'id': row['yuid'],
                    'id_type': 'yandexuid',
                    'segment_name': 'cinema_goers_often',
                }
            elif row['puid']:
                yield {
                    'id': str(row['puid']),
                    'id_type': 'puid',
                    'segment_name': 'cinema_goers_often',
                }


segment_query_template = """
PRAGMA yt.InferSchema = '1';

$cinema_permalinks = (
    SELECT permalink
    FROM `{organization_categories}`
    WHERE 30183 in Yson::ConvertToInt64List(categories_with_parents)
);

$merged_org_visits = (
    SELECT id, id_type, permalink, `date`
    FROM `{org_visits_table}`
    GROUP BY id, id_type, permalink, `date`
);

$frequent_cinema_visitors = (
    SELECT id, id_type, 'cinema_goers_often' AS segment_name
    FROM (
        SELECT
            visits.id AS id,
            visits.id_type AS id_type,
            visits.`date` AS `date`
        FROM $merged_org_visits AS visits
        INNER JOIN $cinema_permalinks AS cinemas
        USING (permalink)
    )
    GROUP BY id, id_type
    HAVING ListLength(AGGREGATE_LIST_DISTINCT(`date`)) >= 3
);

INSERT INTO `{output_table}`
WITH TRUNCATE
SELECT id, id_type, segment_name
FROM `{tickets_table}`
UNION ALL
SELECT id, id_type, segment_name
FROM $frequent_cinema_visitors;
"""


class CinemaGoersOften(RegularSegmentBuilder):
    name_segment_dict = {
        'cinema_goers_often': 11165425,
    }

    keyword = 557

    tickets_number_of_days = 180
    organizations_number_of_days = 180

    def requires(self):
        return {
            'AfishaTicketOrderLog': ExternalInput(
                config.AFISHA_TICKET_ORDER_LOG,
                columns=('yuid', 'puid', 'event_type', 'action_type', 'created_dt'),
            ),
            'OrgDictionary': OrganizationCategoryDictionary(self.date),
            'OrgVisits': LogProcessor(
                CinemaVisitsDayProcessor,
                self.date,
                self.organizations_number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as tickets_table:
            self.yt.run_map(
                partial(ticket_mapper, days=self.tickets_number_of_days, today_date=self.date),
                inputs['AfishaTicketOrderLog'].table,
                tickets_table,
            )

            self.yql.query(
                query_string=segment_query_template.format(
                    organization_categories=inputs['OrgDictionary'].table,
                    org_visits_table=inputs['OrgVisits'].table,
                    tickets_table=tickets_table,
                    output_table=output_path,
                ),
                transaction=self.transaction,
            )
