#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


SUBCATEGORIES_DICT = {
    'baby_toys': [
        'игрушки',
        'наклейки',
        'пластилин',
        'конструктор',
        'погремушка',
        'кукла',
        'паровозик',
    ],

    'food_for_kids': [
        'яйцо с игрушкой',
        'бифидокефир детский',
        'вода детская',
        'творог детский',
        'йогурт детский',
        'каша детская',
        'нектар детский',
        'сок детский',
        'чай детский',
        'шампанское детское',
        'десерт с игрушкой',
        'ряженка детская',
        'кисель детский',
        'яйцо-раскраска',
        'молоко детское',
        'волшебная соломинка',
    ],

    'baby_cosmetics': [
        'мыло детское',
        'присыпка детская',
        'масло детское',
        'детская косметика',
        'крем-пена детская',
        'средства для купания',
        'шампунь детский',
        'крем детский',
        'молочко детское'
    ],

    'baby_other': [
        'дневник',
        'пенал',
        'наклейки',
        'пластилин',
        'конструктор',
        'погремушка',
        'кукла',
        'паровозик'
    ],

    'baby_diapers': [
        'подгузники-трусики',
        'подгузники',
    ],

    'supplies_for_cats': [
        'корм для кошек',
        'корм для котят',
        'лакомство для кошек',
        'когтеточка для кошек',
        'игрушка для кошек',
        'шампунь для кошек',
    ],

    'supplies_for_dogs': [
        'корм для собак',
        'лакомство для собак',
    ],

    'cosmetics': [
        'шампунь для волос',
        'краска для волос',
        'маска для лица',
        'бальзам для волос',
        'крем для рук',
        'лак для волос',
        'бальзам-ополаскиватель',
        'крем для лица',
        'бальзам для губ',
        'мицеллярная вода',
        'лак для ногтей',
        'пудра',
        'тоник для лица',
        'маска для волос',
        'помада',
        'кондиционер для волос',
        'гель для умывания',
        'скраб для ног',
        'очищающие полоски',
        'для ногтей',
        'флюид для век',
        'салфетки для лица',
        'лосьон для снятия макияжа',
        'пенка для умывания',
        'скраб для тела',
        'крем для ног',
        'молочко для тела',
        'карандаш для глаз',
        'гель-шампунь',
        'мусс для волос',
        'мусс для укладки',
        'скраб для лица',
        'сыворотка для волос',
        'гигиеническая помада',
        'масло для волос',
        'карандаш для бровей',
        'лосьон для тела',
        'крем для тела',
        'лосьон для лица',
        'маска для глаз',
        'флюид для умывания',
        'пемза для ног',
        'мусс для лица',
        'бальщам для тела',
        'мусс для умывания',
        'осветлители для волос',
        'скраб для рук',
        'средство для снятия макияжа с глаз',
        'крем для век',
        'ополаскиватель для волос',
        'средство для лица',
        'масло для тела',
        'масло для губ',
        'кондиционер для тела',
        'крем-ополаскиватель',
        'масло для лица',
    ],

    'sweets': [
        'шоколад',
        'печенье',
        'сладкая выпечка',
        'конфеты',
        'мороженое',
        'батончики',
        'жевательная резинка',
        'вафли',
        'пирожные',
        'мармелад',
        'десерт',
        'карамель',
        'драже',
        'торт',
        'пряники',
        'шоколадный батончик',
        'яйцо с игрушкой',
        'зефир',
        'молоко сгущеное',
        'пастила',
        'десерт творожный',
        'выпечка',
        'машмеллоу',
        'полоска песочная',
        'халва',
        'кекс',
        'набор конфет',
        'желе',
        'козинак',
        'паста шоколадная',
        'чак-чак',
        'жевательные конфеты',
        'суфле',
        'пряники мини',
        'нуга',
        'вафли мягкие',
        'варенье',
        'вафельный торт',
        'коллекция изысканных конфет',
        'печенье-сэндвич',
        'сорбет',
        'вафельные рулетики',
        'шоколадный набор',
        'барбарис',
        'пахлава',
        'чурчхела',
        'крепс-десерт',
        'мусс десерт',
        'десерт шоколадный',
    ],

    'household_chemicals': [
        'гель для мытья посуды',
        'губка для посуды',
        'чистящее средство',
        'моющее средство',
        'гель чистящий',
        'тряпка',
        'мыло хозяственное',
        'чистящее средство для туалета',
        'очиститель унитаза',
        'туалетный блок',
        'очиститель',
        'средство для мытья сантехники',
        'насадка для швабры',
        'жироудалитель',
        'перчатки резиновые',
        'средство от засоров',
        'средство для удаления жира',
        'антистатик',
        'средство для кухни',
        'мешок для стирки',
        'салфетки для уборки',
        'щетка для посуды',
        'средство многофункциональное',
        'освежитель для туалета',
        'средство от моли',
        'насадка-тряпка',
        'гель от моли',
        'средство для мытья окон',
        'средство для борьбы с насекомыми',
        'средство для чистки стеклокерамики',
        'очиститель для ванн',
    ],

    'coffee': [
        'кофе растворимый',
        'кофе',
        'кофе в зернах',
        'кофе в капсулах',
        'кофе молотый',
    ],

    'tea': [
        'чай',
        'чай белый',
        'чай в капсулах',
        'чай зеленый',
        'чай травяной',
        'чай фруктовый',
        'чай черный',
        'чайник заварочный',
    ],

    'juice': [
        'сок',
        'сок детский',
        'сок концентрированный',
        'нектар',
    ],

    'water': [
        'вода',
        'вода минеральная',
    ],

    'ice tea': [
        'напиток чайный',
        'чай холодный',
    ],

    'energy drink': [
        'напиток энергетический',
    ],

    'chips': [
        'чипсы',
        'чипсы кукурузные',
    ],

    'snacks': [
        'крекер',
        'снеки',
    ],

    'low-alcohol': [
        'напиток слабоалкогольный',
        'пиво',
    ],

    'cheese': [
        'сыр',
        'сыр копченый',
        'сыр мягкий',
        'сыр плавленый',
        'сыр полутвердый',
        'сыр твердый',
        'сыр творожный',
    ],

    'yogurt': [
        'творожок',
        'йогурт',
        'биойогурт',
        'йогурт греческий',
        'йогурт термостатный',
        'продукт йогуртный',
    ],

    'diary': [
        'молоко',
        'молоко сухое',
        'напиток кисломолочный',
        'напиток молочный',
        'соевое молоко',
        'коктейль молочный',
    ],

    'soda': [
        'напиток газированный',
        'вода газированная',
    ],

    'dishwasher': [
        'жидкое средство для посудомойки',
        'капсулы для посудомоечных машин',
        'ополаскиватель для посудомоечной машины',
        'освежитель для посудомоечных машин',
        'очиститель для посудомоечных машин',
        'порошок для посудомоечных машин',
        'соль для посудомоечной машины',
        'таблетки для посудомоечных машин',
    ],

    'hair': [
        'бальзам-ополаскиватель для волос',
        'кондиционер для волос',
        'маска для волос',
        'крем для волос',
        'cкраб для волос и кожи головы',
        'cпрей для волос',
        'cыворотка для волос',
        'тоник для волос',
        'масло для волос',
        'гель-шампунь',
        'гель-шампунь мужской',
    ],

    'shaving': [
        'бальзам после бритья',
        'гель для бритья',
        'гель после бритья',
        'кассеты для бритья',
        'крем для бритья',
        'крем после бритья',
        'лосьон после бритья',
        'набор для бритья',
        'пена для бритья',
        'бритвенный станок',
    ],

    'laundary': [
        'cтиральный порошок',
        'капсулы для стирки',
        'средство для стирки',
        'отбеливатель',
        'пятновыводитель',
        'гель для стирки',
        'кондиционер для белья',
        'средство для стиральной машины',
    ],
}

CATEGORIES_DICT = {
    'goods for children': list(set(
        SUBCATEGORIES_DICT['baby_cosmetics']
        + SUBCATEGORIES_DICT['baby_toys']
        + SUBCATEGORIES_DICT['baby_other']
        + SUBCATEGORIES_DICT['baby_diapers']
        + SUBCATEGORIES_DICT['food_for_kids']
    )),
    'baby_diapers': SUBCATEGORIES_DICT['baby_diapers'],
    'food_for_kids': SUBCATEGORIES_DICT['food_for_kids'],
    'baby_toys': SUBCATEGORIES_DICT['baby_toys'],
    'supplies_for_cats': SUBCATEGORIES_DICT['supplies_for_cats'],
    'supplies_for_dogs': SUBCATEGORIES_DICT['supplies_for_dogs'],
    'coffee': SUBCATEGORIES_DICT['coffee'],
    'cosmetics_mass_market': SUBCATEGORIES_DICT['cosmetics'],
    'sweets': SUBCATEGORIES_DICT['sweets'],
    'household_chemicals':
        SUBCATEGORIES_DICT['household_chemicals']
        + SUBCATEGORIES_DICT['dishwasher']
        + SUBCATEGORIES_DICT['laundary'],
    'tea': SUBCATEGORIES_DICT['tea'],
    'juice': SUBCATEGORIES_DICT['juice'],
    'water': SUBCATEGORIES_DICT['water'],
    'ice tea': SUBCATEGORIES_DICT['ice tea'],
    'energy drink': SUBCATEGORIES_DICT['energy drink'],
    'chips': SUBCATEGORIES_DICT['chips'],
    'low-alcohol': SUBCATEGORIES_DICT['low-alcohol'],
    'cheese': SUBCATEGORIES_DICT['cheese'],
    'yogurt': SUBCATEGORIES_DICT['yogurt'],
    'diary': SUBCATEGORIES_DICT['diary'],
    'soda': SUBCATEGORIES_DICT['soda'],
    'snacks': SUBCATEGORIES_DICT['snacks'],
    'dishwasher': SUBCATEGORIES_DICT['dishwasher'],
    'hair': SUBCATEGORIES_DICT['hair'],
    'shaving': SUBCATEGORIES_DICT['shaving'],
    'laundary': SUBCATEGORIES_DICT['laundary'],
}

segment_query_template = u"""
$segment_name_with_categories = (
    SELECT
        attributes.item_id AS item_id,
        segment_names_with_categories.segment_name AS segment_name,
        IF (segment_names_with_categories.segment_name == 'juice', attributes.attr_14) AS volume
    FROM `{attributes_table}` AS attributes
    INNER JOIN `{segment_names_with_categories}` AS segment_names_with_categories
    ON String::ToLower(attributes.attr_482) == segment_names_with_categories.category
);

$receipt_edadeal_device_ids = (
    SELECT
        receipts.device_id AS device_id,
        segment_name_with_categories.segment_name AS segment_name,
        IF (segment_name_with_categories.segment_name == 'juice', receipts.price * 1.0 / 100) AS price,
        IF (segment_name_with_categories.segment_name == 'juice', segment_name_with_categories.volume * 1.0 / 1000) AS volume
    FROM `{receipts_table}` AS receipts
    INNER JOIN $segment_name_with_categories AS segment_name_with_categories
    ON receipts.id == segment_name_with_categories.item_id
    WHERE receipts.check_date > '{edadeal_start_date}'
);

$edadeal_segments = (
    SELECT device_id, segment_name
    FROM $receipt_edadeal_device_ids

    UNION ALL

    SELECT device_id, 'premium juice' AS segment_name
    FROM $receipt_edadeal_device_ids
    WHERE segment_name == 'juice'
        AND price / volume >= 80
        AND volume >= 0.5
);

$receipt_edadeal_mm_device_ids = (
    SELECT
        matching.DeviceID AS id,
        'mm_device_id' AS id_type,
        edadeal_segments.segment_name AS segment_name
    FROM $edadeal_segments AS edadeal_segments
    INNER JOIN `{edadeal_device_id_to_mm_device_id_table}` AS matching
    ON edadeal_segments.device_id == matching.edadealDUID
);

INSERT INTO `{output_path}` WITH TRUNCATE
SELECT id, id_type, segment_name
FROM $receipt_edadeal_mm_device_ids
GROUP BY id, id_type, segment_name
"""

CATEGORIES_TO_SEGMENT_NAME_TABLE = os.path.join(config.LONG_TERM_DATA_FOLDER, 'edadeal_categories_to_segment_names')


class OfflinePurchases(RegularSegmentBuilder):
    keyword = 547
    number_of_days = 7 * 8

    name_segment_dict = {
        'goods for children': 1164,
        'baby_diapers': 1165,
        'food_for_kids': 1166,
        'baby_toys': 1167,
        'supplies_for_cats': 1168,
        'supplies_for_dogs': 1169,
        'coffee': 1170,
        'cosmetics_mass_market': 1171,
        'household_chemicals': 1172,
        'sweets': 1173,
        'tea': 1248,
        'juice': 1346,
        'premium juice': 1351,
        'water': 1340,
        'ice tea': 1348,
        'energy drink': 1350,
        'chips': 1349,
        'snacks': 1345,
        'low-alcohol': 1344,
        'cheese': 1347,
        'yogurt': 1342,
        'diary': 1343,
        'soda': 1341,
        'dishwasher': 2090,
        'hair': 2098,
        'shaving': 2099,
        'laundary': 2100,
    }

    def requires(self):
        return {
            'edadeal_receipts_scans': ExternalInput(os.path.join(config.EDADEAL_RECEIPTS_SCANS, self.date)),
            'edadeal_item_attributes': ExternalInput(os.path.join(config.EDADEAL_ITEM_ATTRIBUTES, '0000-00-00')),
            'edadeal_devices': ExternalInput(os.path.join(config.EDADEAL_DEVICES, '0000-00-00')),
        }

    def prepare_categories_table(self):
        self.yt.create_empty_table(
            CATEGORIES_TO_SEGMENT_NAME_TABLE,
            schema={
                'segment_name': 'string',
                'category': 'string',
            },
        )

        segment_names_with_categories_list = []

        for key, values_list in CATEGORIES_DICT.iteritems():
            segment_names_with_categories_list.extend(
                [{'segment_name': key, 'category': value} for value in values_list]
            )

        self.yt.write_table(
            CATEGORIES_TO_SEGMENT_NAME_TABLE,
            segment_names_with_categories_list,
        )

    def build_segment(self, inputs, output_path):
        self.prepare_categories_table()

        self.yql.query(
            query_string=segment_query_template.format(
                receipts_table=inputs['edadeal_receipts_scans'].table,
                attributes_table=inputs['edadeal_item_attributes'].table,
                edadeal_device_id_to_mm_device_id_table=inputs['edadeal_devices'].table,
                edadeal_start_date=date_helpers.get_date_from_past(self.date, self.number_of_days),
                output_path=output_path,
                segment_names_with_categories=CATEGORIES_TO_SEGMENT_NAME_TABLE,
            ),
            transaction=self.transaction,
        )
