import os
from functools import partial

import luigi

from crypta.lib.python import templater
from crypta.profile.lib import date_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput, YtNodeAttributeTarget, BaseYtTask
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


marker_cpa_click_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    yandex_uid AS id,
    'yandexuid' AS id_type,
    'eshoppers' AS segment_name
FROM (
    SELECT yandex_uid
    FROM `{input_table}`
    WHERE yandex_uid IS NOT NULL AND yandex_uid != ''
)
GROUP BY yandex_uid
"""

eshoppers_query = """
$segment = (
    SELECT
        yandexuid AS id,
        'yandexuid' AS id_type,
        'eshoppers' AS segment_name,
    FROM `{{ecom_table}}`
    WHERE order_date >= "{{start_date}}" AND yandexuid IS NOT NULL

    UNION ALL

    SELECT * FROM `{{market_cpa_clicks_log}}`
);

INSERT INTO `{{output_table}}` WITH TRUNCATE
SELECT DISTINCT id, id_type, segment_name
FROM $segment
ORDER BY id, id_type, segment_name
"""


class ProcessMarketCpaClickLogForEshoppers(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.MARKET_CPA_CLICKS_LOG_FOLDER,
                self.date,
            ),
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            marker_cpa_click_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


def online_purchases_reducer(key, rows):
    ans = dict(key)
    if not ans['id']:
        return

    ans['price'] = sum(row['price'] for row in rows)
    ans['id_type'] = 'yandexuid'
    ans['segment_name'] = 'eshoppers'
    yield ans


class ProcessOnlinePurchases(BaseYtTask):
    date = luigi.Parameter()
    task_group = 'coded_segments'

    def __init__(self, date):
        super(ProcessOnlinePurchases, self).__init__(date)
        self.last_processed_table = None
        if self.yt.exists(config.PROCESSED_ONLINE_PURCHASES):
            self.last_processed_table = self.yt.get_attribute(
                config.PROCESSED_ONLINE_PURCHASES,
                'last_processed_table',
                None,
            )

        self.last_available_table = max(self.yt.list(config.ONLINE_PURCHASES_MONTHLY_FOLDER))
        self.input_tables = []

    def requires(self):
        self.input_tables = []
        for node in self.yt.list(config.ONLINE_PURCHASES_MONTHLY_FOLDER):
            if self.last_processed_table < node <= self.last_available_table:
                self.input_tables.append(os.path.join(config.ONLINE_PURCHASES_MONTHLY_FOLDER, node))

        return [ExternalInput(table) for table in self.input_tables]

    def output(self):
        return YtNodeAttributeTarget(
            path=config.PROCESSED_ONLINE_PURCHASES,
            attribute_name='last_processed_table',
            attribute_value=self.last_available_table,
        )

    def run(self):
        with self.yt.Transaction():
            if not self.yt.exists(config.PROCESSED_ONLINE_PURCHASES):
                self.yt.create_empty_table(
                    config.PROCESSED_ONLINE_PURCHASES,
                    schema={
                        'id': 'string',
                        'id_type': 'string',
                        'segment_name': 'string',
                        'order_id': 'string',
                        'price': 'double',
                    },
                )

            self.yt.run_map_reduce(
                None,
                online_purchases_reducer,
                [self.yt.TablePath(
                    table,
                    columns=('id', 'order_id', 'price'),
                    rename_columns={'yandexuid': 'id'},
                ) for table in self.input_tables],
                self.yt.TablePath(self.output().path, append=True),
                reduce_by=['id', 'order_id'],
            )

            self.yt.set_attribute(
                self.output().path,
                'last_processed_table',
                self.last_available_table,
            )


class Eshoppers(RegularSegmentBuilder):
    name_segment_dict = {
        'eshoppers': (547, 1050),
    }

    number_of_days = 90

    def requires(self):
        return {
            'MarketCpaClicksLog': LogProcessor(
                ProcessMarketCpaClickLogForEshoppers,
                self.date,
                self.number_of_days,
            ),
            'EShoppersByEcom': ExternalInput(config.ONLINE_PURCHASES_AGGREGATED_LOG),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            templater.render_template(eshoppers_query, strict=True, vars=dict(
                ecom_table=inputs['EShoppersByEcom'].table,
                market_cpa_clicks_log=inputs['MarketCpaClicksLog'].table,
                start_date=date_helpers.get_date_from_past(self.date, years=1),
                output_table=output_path,
            )),
            transaction=self.transaction,
        )


def filter_reducer(key, rows, last_date):
    max_price = None

    for row in rows:
        order_date = row['order_id'].split(':')[0]
        if order_date >= last_date:
            max_price = max(max_price, row['price'])

    if max_price >= 5000:
        ans = dict(key)
        ans['segment_name'] = 'eshoppers_more_than_5K'
        yield ans


class EshoppersMoreThan5K(RegularSegmentBuilder):
    name_segment_dict = {
        'eshoppers_more_than_5K': (557, 8843842),
    }

    def requires(self):
        return ProcessOnlinePurchases(self.date)

    def build_segment(self, inputs, output_path):
        self.yt.run_map_reduce(
            None,
            partial(
                filter_reducer,
                last_date=date_helpers.get_date_from_past(self.date, 180),
            ),
            self.yt.TablePath(inputs.path, columns=['id', 'id_type', 'price', 'order_id']),
            output_path,
            reduce_by=['id', 'id_type'],
        )
