#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
from functools import partial

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInputDate
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


def get_date_without_year(date):
    return date.split('-', 1)[1]


def segment_mapper(row, left_boundary, right_boundary):
    if row['birth_date'] is None:
        return

    date_without_year = get_date_without_year(row['birth_date'])
    if date_without_year == '01-01':  # anomaly date
        return

    if left_boundary <= date_without_year <= right_boundary:
        yield {
            'id': row['id'],
            'id_type': row['id_type'],
            'segment_name': 'have_birthday_soon',
        }


class HaveBirthdaySoon(RegularSegmentBuilder):
    name_segment_dict = {
        'have_birthday_soon': (547, 2134),
    }

    def requires(self):
        return ExternalInputDate(
            os.path.join(
                config.SOCDEM_STORAGE_YT_DIR,
                'puid',
                'passport',
            ),
            self.date,
        )

    def build_segment(self, inputs, output_path):
        left_boundary = get_date_without_year(date_helpers.get_date_from_past(self.date, 5))
        right_boundary = get_date_without_year(date_helpers.get_date_from_past(self.date, -8))

        self.yt.run_map(
            partial(segment_mapper, left_boundary=left_boundary, right_boundary=right_boundary),
            self.yt.TablePath(self.input().table, columns=['id', 'id_type', 'birth_date']),
            output_path,
        )
