#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import AttributeExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


day_processor_query = u"""
$kid_related_words = AsList(
    "болеть", "горло", "детсад", "живот", "животик", "здравие", "здоровье", "зуб", "игра", "игрушка",
    "интерес", "интересовать", "кашель", "книга", "книжка", "колыбельный", "конкурс", "косичка", "кружок",
    "кукла", "лечить", "машинка", "молоко", "мультфильм", "насморк", "нос", "песенка", "писать", "питание", "плакать",
    "плач", "подарок", "прививка", "привязанность", "прическа", "развивать", "развитие", "расстройство", "режим",
    "рисовать", "садик", "самостоятельный", "секция", "сок", "стрижка", "сыпь", "телефон", "температура", "улица",
    "успокоить", "ухо", "художество", "чтение"
);

$rules = AsList(
".* девочка | малыш | мальчик | младенец | ребенок .*",
".* детский .* сад .*",
".* " || String::JoinFromList($kid_related_words, " | ") || " .*"
);

$match = Hyperscan::MultiMatch(String::JoinFromList($rules, '\n'));

$m = ($lemmas) -> {{
    RETURN $match(' ' || String::JoinFromList(ListSort($lemmas), '  ') || ' ');
}};

$is_condition_satisfied = ($lemmas) -> {{
    $mm = $m($lemmas);
    RETURN $mm.0 AND ($mm.2 OR $mm.1);
}};


INSERT INTO `{output_table}` WITH TRUNCATE
SELECT DISTINCT
    CAST(yandexuid AS String) AS id,
    'yandexuid' AS id_type,
     TableName() AS `date`
FROM `{input_table}`
WHERE yandexuid != 0 AND $is_condition_satisfied(Yson::ConvertToStringList(lemmas))
"""


class ProcessedReqansForWithChildrenAddition(DayProcessor):
    def requires(self):
        return AttributeExternalInput(
            os.path.join(
                config.REQANS_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


with_children_query = """
$with_children_ids = (
    SELECT id, id_type
    FROM `{dit_all_info_table}`
    WHERE children
);

$with_children_yandexuid = (
    SELECT id, id_type
    FROM `{reqans_table}`
    GROUP BY id, id_type
    HAVING ListLength(AGGREGATE_LIST_DISTINCT(`date`)) > 1
);

$with_children_crypta_id = (
    SELECT matching.cryptaId AS crypta_id
    FROM $with_children_ids AS with_children
    INNER JOIN `{id_to_crypta_id_table}` AS matching
    USING (id, id_type)
);

$with_children_crypta_id_hhid = (
    SELECT matching.hhid AS hhid
    FROM $with_children_crypta_id AS with_children
    INNER JOIN `{crypta_id_to_hhid_table}` AS matching
    USING (crypta_id)
);

$with_children_yandexuid_hhid = (
    SELECT matching.hhid AS hhid
    FROM $with_children_yandexuid AS with_children
    INNER JOIN `{yandexuid_to_hhid_table}` AS matching
    ON with_children.id == matching.yuid
);

$segment_hhids = (
    SELECT hhid
    FROM (
        SELECT *
        FROM $with_children_crypta_id_hhid
        UNION ALL
        SELECT *
        FROM $with_children_yandexuid_hhid
    )
    GROUP BY hhid
);

$segment_yandexuid_to_hhid = (
    SELECT
        matching.yuid AS id,
        'yandexuid' AS id_type
    FROM $segment_hhids AS segment
    INNER JOIN `{hhid_to_yandexuid_table}` AS matching
    USING (hhid)
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, 'with_children_of_unknown_age' AS segment_name
FROM (
    SELECT *
    FROM $segment_yandexuid_to_hhid
    UNION ALL
    SELECT *
    FROM $with_children_yandexuid
)
GROUP BY id, id_type
"""


class WithChildrenOfUnknownAge(RegularSegmentBuilder):
    name_segment_dict = {
        'with_children_of_unknown_age': (547, 1918),
    }

    number_of_days = 35

    def requires(self):
        return {
            'ProcessedReqans': LogProcessor(
                ProcessedReqansForWithChildrenAddition,
                self.date,
                self.number_of_days,
            )
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            with_children_query.format(
                reqans_table=inputs['ProcessedReqans'].table,
                dit_all_info_table=config.DIT_INFO_TABLE,
                id_to_crypta_id_table=config.VERTICES_NO_MULTI_PROFILE,
                crypta_id_to_hhid_table=config.HOUSEHOLD_CRYPTA_ID_TO_HHID,
                yandexuid_to_hhid_table=config.HOUSEHOLD_REVERSED_TABLE,
                hhid_to_yandexuid_table=config.HOUSEHOLD_ENRICH_TABLE,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
