#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.lib import date_helpers

from crypta.lib.python import templater
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder

insurance_type_to_catalogia_id = {
    'osago': 200027663,
    'kasko': 200027658,
}

update_longterm_data_query = """
$new_data = (
    SELECT
        CAST(yandexuid AS String) AS id,
        'yandexuid' AS id_type,
        DateTime::Format("%Y-%m-%d")(Datetime::FromSeconds(CAST(`timestamp` AS Uint32))) AS `date`,
        category
    FROM RANGE(`{input_table_prefix}`, `{start_date}`, `{end_date}`)
    WHERE yandexuid != 0 AND category IN ({osago_catalogia_id}, {kasko_catalogia_id})
);

INSERT INTO `{longterm_data_table}` WITH TRUNCATE
SELECT id, id_type, `date`, category
FROM (
    SELECT *
    FROM $new_data
        UNION ALL
    SELECT *
    FROM `{longterm_data_table}`
)
GROUP BY id, id_type, `date`, category
ORDER BY `date`;
"""

segment_query_template = """
INSERT INTO `{{output_table}}`
WITH TRUNCATE

{% for segment_parameters in segments_parameters %}
{% if loop.index != 1 %}
        UNION ALL
{% endif %}
SELECT
    id,
    id_type,
    '{{segment_parameters.segment_name}}' AS segment_name
FROM `{{longterm_data_table}}`
WHERE `date` >= '{{segment_parameters.start_date}}'
    AND `date` <= '{{segment_parameters.end_date}}'
    AND category == {{segment_parameters.catalogia_id}}
{% endfor %}
"""


class InterestedInCompulsoryAutoInsurance(RegularSegmentBuilder):
    name_segment_dict = {
        'osago_10_months_ago': (557, 19430527),
        'osago_12_months_ago': (557, 16478659),
        'kasko_11_months_ago': (557, 19877349),
    }

    def __init__(self, date):
        super(InterestedInCompulsoryAutoInsurance, self).__init__(date)
        self.longterm_data_table = os.path.join(config.LONG_TERM_DATA_FOLDER, 'car_insurance')
        self.last_processed_date = self.yt.get_attribute(self.longterm_data_table, 'last_processed_date')

    def requires(self):
        return {
            'new_data': [
                ExternalInput(os.path.join(config.BB_PARSED_DIR, date))
                for date in date_helpers.generate_date_strings(
                    date_helpers.get_tomorrow(self.last_processed_date),
                    self.date,
                )
            ],
            'longterm_data': ExternalInput(self.longterm_data_table),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            update_longterm_data_query.format(
                input_table_prefix=config.BB_PARSED_DIR,
                start_date=date_helpers.get_tomorrow(self.last_processed_date),
                end_date=self.date,
                longterm_data_table=self.input()['longterm_data'].table,
                osago_catalogia_id=insurance_type_to_catalogia_id['osago'],
                kasko_catalogia_id=insurance_type_to_catalogia_id['kasko'],
            ),
            transaction=self.transaction,
        )

        self.yt.set_attribute(
            self.input()['longterm_data'].table,
            'last_processed_date',
            self.date,
        )

        segments_parameters = []
        for insurance_type, months_back in (('osago', 12), ('osago', 10), ('kasko', 11)):
            segments_parameters.append({
                'segment_name': '{}_{}_months_ago'.format(insurance_type, months_back),
                'start_date': date_helpers.get_date_from_past(self.date, months=months_back, days=14),
                'end_date': date_helpers.get_date_from_past(self.date, months=months_back - 1, days=14),
                'catalogia_id': insurance_type_to_catalogia_id[insurance_type],
            })
        segments_query = templater.render_template(
            template_text=segment_query_template,
            vars={
                'segments_parameters': segments_parameters,
                'longterm_data_table': self.input()['longterm_data'].table,
                'output_table': output_path,
            },
        )

        self.yql.query(
            query_string=segments_query,
            transaction=self.transaction,
        )
