#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


daily_query = """
$types_3g = AsSet(
    'HSPA', 'HSPA+', 'UMTS', 'HSDPA', 'HSUPA', 'WCDMA', 'CDMA', '3G',
);

$types_4g = AsSet(
    'LTE', '4G',
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    DeviceID AS id,
    segment_name
FROM (
    SELECT
        DeviceID,
        CASE
            WHEN NetworkType IN $types_3g THEN '3g'
            WHEN NetworkType IN $types_4g THEN '4g'
            ELSE NULL
        END AS segment_name
    FROM `{input_table}`
)
WHERE segment_name IS NOT NULL
GROUP BY DeviceID, segment_name;
"""


class ProcessAppMetricaForConnectionType(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.APP_METRICA_YANDEX_DAILY,
                self.date,
            )
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            daily_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


connection_type_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    'mm_device_id' AS id_type,
    CASE
        WHEN types == AsSet('3g') THEN '3g'
        WHEN types == AsSet('4g') THEN '4g'
        ELSE '3g_4g'
    END AS segment_name
FROM (
    SELECT
        id,
        ToSet(AGGREGATE_LIST_DISTINCT(segment_name)) AS types
    FROM `{input_table}`
    GROUP BY id
);
"""


class ConnectionType(RegularSegmentBuilder):
    name_segment_dict = {
        '3g': (557, 17823841),
        '4g': (557, 17823853),
        '3g_4g': (557, 17823847),
    }

    number_of_days = 35

    def requires(self):
        return {
            'AppMetrica': LogProcessor(
                ProcessAppMetricaForConnectionType,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            connection_type_query.format(
                input_table=inputs['AppMetrica'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
