#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


console_buyers_query = """
$console_categories = AsList(
    22144, 10224, 22083, 1509, 10214, 10092, 10090,
    23147, 23148, 50, 10094, 24527, 24528, 21586,
    10223, 22143
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    yandex_uid AS id,
    'yandexuid' AS id_type,
    'console_gamers' AS segment_name
FROM (
    SELECT yandex_uid
    FROM `{input_table}`
    WHERE yandex_uid IS NOT NULL AND yandex_uid != '' AND category_id IN $console_categories
)
GROUP BY yandex_uid
"""


class ProcessMarketCpaClickLogForConsoleGamers(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.MARKET_CPA_CLICKS_LOG_FOLDER,
                self.date,
            )
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            console_buyers_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


def extract_yandexuids_from_yuid_with_all(row):
    if row['ua'] and row['ip_activity_type']:
        ua = row['ua'].lower()
        if row['ip_activity_type'] == 'active' and \
                any(word in ua for word in ['playstation', 'xbox', 'nintendo']):
            yield {
                'id': row['id'],
                'id_type': row['id_type'],
                'segment_name': 'console_gamers',
            }


console_yandexuid_query = """
$re = Hyperscan::Grep('(?i)playstation|xbox|nintendo');

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    'console_gamers' AS segment_name
FROM `{yuid_with_all_info_table}`
WHERE ip_activity_type == 'active' AND $re(ua);
"""


class ConsoleGamers(RegularSegmentBuilder):
    name_segment_dict = {
        'console_gamers': (547, 1966),
    }

    number_of_days = 30

    def requires(self):
        return {
            'MarketCpaClicksLog': LogProcessor(
                ProcessMarketCpaClickLogForConsoleGamers,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as console_yuids:
            self.yql.query(
                console_yandexuid_query.format(
                    yuid_with_all_info_table=config.YUID_WITH_ALL_INFO_TABLE,
                    output_table=console_yuids,
                ),
                transaction=self.transaction,
            )

            self.yt.unique(
                [inputs['MarketCpaClicksLog'].table,
                 console_yuids],
                output_path,
                unique_by=['id', 'id_type', 'segment_name'],
            )
