#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import (
    LaLParams,
    RegularSegmentBuilder,
)
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


select_clients_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    clientid AS ClientID,
    SUM_IF(value_actual, selecttype IN (0, 5, 17)) AS search_cost,
    SUM_IF(value_actual, selecttype NOT IN (0, 5, 17)) AS yan_cost,
    SUM_IF(value_actual, contexttype == 2) AS retargeting_cost,
    SUM_IF(value_actual, producttype == 'video-creative') AS cpc_video_cost,
    SUM_IF(value_actual, producttype == 'audio-creative-reach') AS audio_cost,
    -- https://wiki.yandex-team.ru/direct/products/reachproduct/video/video-types/
    SUM_IF(value_actual, producttype IN ('video-creative', 'auto-video-direct', 'video-motion', 'video-creative-reach')) AS all_video_cost,
    SUM_IF(value_actual, campaign_type IN ('cpm_banner', 'cpm_deals', 'cpm_yndx_frontpage', 'cpm_price'))
    AS reach_cost,
FROM `{input_table}`
WHERE date IS NOT NULL AND clientid IS NOT NULL
GROUP BY clientid;
"""

direct_campaigns_query = u"""
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$goals_strategy = (
    "Автобюджет: Оптимизация количества конверсий (CPA на всю кампанию)",
    "Автобюджет: Оптимизация количества конверсий (CPA на каждый фильтр)",
    "Автобюджет: Оптимизация конверсий (недельный бюджет)",
    "Автобюджет: Средняя рентабельность инвестиций",
    "Автобюджет: Средняя цена конверсии",
    "Автобюджет: Средняя цена установки приложения",
);


INSERT INTO `{output_table}` WITH TRUNCATE
SELECT client_id
FROM `{campaigns_cube}`
WHERE
    campaign_strategy_name IN $goals_strategy AND
    shows > 0
GROUP BY client_id AS client_id;
"""

build_segment_query = u"""
$metrica_usage = (
    SELECT
        client_id,
        MAX_BY(metrika_orders, cube_date) as last_metrika_orders
    FROM RANGE(`{metrica_usage}`, `{start_date}`, `{end_date}`)
    GROUP BY client_id
);

$campaigns = (
    SELECT ClientID
    FROM RANGE(`{capmaigns_cube}`, `{start_date}`, `{end_date}`) as a
    LEFT ONLY JOIN `{campaigns}` as b
    USING(client_id)
    WHERE
        campaign_type IN ("text", "performance", "mobile_content", "dynamic")
        AND shows > 0
        AND metrika_orders IS NOT NULL
    GROUP BY a.client_id as ClientID
);

$grouped_by_client_id = (
    SELECT
        ClientID,
        SUM(search_cost) ?? 0 AS search_cost,
        SUM(yan_cost) ?? 0 AS yan_cost,
        SUM(retargeting_cost) ?? 0 AS retargeting_cost,
        SUM(cpc_video_cost) ?? 0 AS cpc_video_cost,
        SUM(all_video_cost) ?? 0 AS all_video_cost,
        SUM(audio_cost) ?? 0 AS audio_cost,
        SUM(reach_cost) ?? 0 AS reach_cost,
    FROM `{input_table}`
    GROUP BY ClientID
);

$segment_puid = (
    SELECT
        CAST(direct_users.uid as String) AS id,
        'puid' AS id_type,
        segment.segment_name AS segment_name
    FROM (
        SELECT ClientID, 'search_not_yan_users' as segment_name
        FROM $grouped_by_client_id
        WHERE yan_cost == 0 AND search_cost > 0

        UNION ALL

        SELECT ClientID, 'yan_not_retargeting_users' as segment_name
        FROM $grouped_by_client_id
        WHERE yan_cost > 0 AND retargeting_cost == 0

        UNION ALL

        SELECT ClientID, 'yan_not_cpc_video_users' as segment_name
        FROM $grouped_by_client_id
        WHERE yan_cost > 0 AND cpc_video_cost == 0

        UNION ALL

        SELECT ClientID, 'video_users' as segment_name
        FROM $grouped_by_client_id
        WHERE all_video_cost > 0

        UNION ALL

        SELECT ClientID, 'not_audio_users' as segment_name
        FROM $grouped_by_client_id
        WHERE audio_cost == 0

        UNION ALL

        SELECT ClientID, 'reach_not_audio_users' as segment_name
        FROM $grouped_by_client_id
        WHERE audio_cost == 0 AND reach_cost > 0

        UNION ALL

        SELECT all_types.cid as ClientID, 'not_geo_users' as segment_name FROM `{direct_geo}` as all_types
        LEFT ONLY JOIN (
            SELECT cid FROM `{direct_geo}`
            WHERE adgroup_type == 'cpm_geoproduct'
            GROUP BY cid) as geo
        USING(cid)
        GROUP BY all_types.cid

        UNION ALL

        SELECT order_info.ClientID as ClientID, 'social_networks_domains' as segment_name
        FROM `{oder_info}` as order_info
        LEFT SEMI JOIN (
            SELECT ServiceOrderID FROM `{billing}`
            WHERE
                Domain ILIKE "%vk.com%"
                OR Domain ILIKE "%fb.com%"
                OR Domain ILIKE "%facebook.com%"
                OR Domain ILIKE "%instagram.com%"
                OR Domain ILIKE "ok.ru%"
                OR Domain ILIKE "%//ok.ru%"
            GROUP BY ServiceOrderID
            ) as billing
        ON order_info.ExportID == billing.ServiceOrderID
        GROUP BY order_info.ClientID

        UNION ALL

        SELECT
            client_id as ClientID,
            'not_metrica_users' as segment_name
        FROM $metrica_usage
        WHERE last_metrika_orders IS NULL

        UNION ALL

        SELECT
            client_id as ClientID,
            'campaign_without_metrica' as segment_name
        FROM $metrica_usage
        WHERE last_metrika_orders == 0

        UNION ALL

        SELECT
            ClientID,
            'strategy_without_metrica' as segment_name
        FROM $campaigns

    ) AS segment
    INNER JOIN `{direct_users_table}` AS direct_users
    USING (ClientID)
    GROUP BY direct_users.uid, segment.segment_name
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, segment_name
FROM $segment_puid
WHERE segment_name IN ('yan_not_cpc_video_users', 'yan_not_retargeting_users', 'search_not_yan_users',
                       'not_audio_users', 'not_geo_users', 'social_networks_domains', 'not_metrica_users',
                       'reach_not_audio_users', 'campaign_without_metrica', 'strategy_without_metrica')

UNION ALL

SELECT
    CAST(uid AS String) AS id,
    'puid' AS id_type,
    'direct_users' AS segment_name
FROM `{direct_users_table}`
;

INSERT INTO `{sample_table}` WITH TRUNCATE
SELECT
    CAST(matching.yandexuid AS String) AS yandexuid,
    sample.segment_name AS segment_name
FROM(
    SELECT
        id,
        id_type,
        segment_name
    FROM $segment_puid
    WHERE segment_name == 'video_users'

    UNION ALL

    SELECT
        CAST(uid AS String) AS id,
        'puid' AS id_type,
        'direct_users' AS segment_name
    FROM `{direct_users_table}`
) AS sample
INNER JOIN `{indevice_yandexuid_matching}` AS matching
USING (id, id_type);
"""


class ProcessedSessionsForDirectProductUsers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join('//statbox/cube/daily/product_money/v1', self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            select_clients_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class ProcessedDirectCampaignsForDirectProductUsers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.DIRECT_CAMPAIGNS_CUBE, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            direct_campaigns_query.format(
                campaigns_cube=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class DirectProductUsers(RegularSegmentBuilder):

    name_segment_dict = {
        'search_not_yan_users': (549, 1277),
        'yan_not_retargeting_users': (549, 1276),
        'yan_not_cpc_video_users': (549, 1278),
        'not_audio_users': (557, 12560353),
        'not_geo_users': (557, 12597940),
        'social_networks_domains': (557, 12585274),
        'not_metrica_users': (557, 12597877),
        'reach_not_audio_users': (557, 12859987),
        'direct_users': (549, 1631),
        'campaign_without_metrica': (557, 17239699),
        'strategy_without_metrica': (557, 17337258),
    }

    number_of_days = 35 * 3

    def requires(self):
        return {
            'ProcessedDirectCube': LogProcessor(
                ProcessedSessionsForDirectProductUsers,
                date_helpers.get_date_from_past(self.date, days=2),
                self.number_of_days,
            ),
            'MetricaUsage': ExternalInput(os.path.join(config.DIRECT_CLIENTS_CUBE, date_helpers.get_date_from_past(self.date, days=2))),
            'DirectUsers': ExternalInput(config.DIRECT_USERS),
            'DirectGeo': ExternalInput(config.DIRECT_GEO),
            'OrderInfo': ExternalInput('//home/yabs/dict/OrderInfo'),
            'BillingOrderDomains': ExternalInput(config.DIRECT_BILLING_ORDER_DOMAINS),
            'Campaigns': LogProcessor(
                ProcessedDirectCampaignsForDirectProductUsers,
                date_helpers.get_date_from_past(self.date, days=2),
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as sample_table:
            self.yql.query(
                build_segment_query.format(
                    input_table=inputs['ProcessedDirectCube'].table,
                    direct_users_table=inputs['DirectUsers'].table,
                    direct_geo=inputs['DirectGeo'].table,
                    oder_info=inputs['OrderInfo'].table,
                    billing=inputs['BillingOrderDomains'].table,
                    indevice_yandexuid_matching=config.INDEVICE_YANDEXUID,
                    output_table=output_path,
                    sample_table=sample_table,
                    start_date=date_helpers.get_date_from_past(self.date, self.number_of_days),
                    end_date=self.date,
                    metrica_usage=config.DIRECT_CLIENTS_CUBE,
                    campaigns=inputs['Campaigns'].table,
                    capmaigns_cube=config.DIRECT_CAMPAIGNS_CUBE,
                ),
                transaction=self.transaction,
            )

            self.prepare_samples_for_lal(
                input_table=sample_table,
                id_field='yandexuid',
                lals_params=[
                    LaLParams(
                        name='video_users',
                        id=1403,
                        type='lal_internal',
                        coverage=3000000,
                        include_input=False,
                    ),
                    LaLParams(
                        name='direct_users',
                        id=19524701,
                        type='audience_segments',
                        coverage=1000000,
                        include_input=False,
                    )
                ],
            )
