#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


day_processor_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, card_id, MAX(card_price) AS card_price
FROM (
    SELECT
        user_yandex_uid AS id,
        'yandexuid' AS id_type,
        card_id,
        CAST(card_price AS Double) AS card_price
    FROM `{input_table}`
    WHERE component == 'card_view'
        AND CAST(user_yandex_uid AS Uint64) IS NOT NULL
        AND CAST(user_yandex_uid AS Uint64) != 0
        AND CAST(card_id AS Uint64) IS NOT NULL
        AND CAST(card_id AS Uint64) != 0
        AND CAST(card_price AS Double) IS NOT NULL
        AND CAST(card_price AS Double) > 0
)
GROUP BY id, id_type, card_id
"""


class ProcessedAutoFrontLogForExpensiveCarCustomers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.AUTORU_FRONT_LOG_FOLDER, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = """
$prices = (
    SELECT id, id_type, card_id, MAX(card_price) AS card_price
    FROM `{input_table}`
    GROUP BY id, id_type, card_id
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, 'expensive_car_customers_1mln' AS segment_name
FROM $prices
GROUP BY id, id_type
HAVING MEDIAN(card_price) >= 1000000

UNION ALL

SELECT id, id_type, 'expensive_car_customers_2mln' AS segment_name
FROM $prices
GROUP BY id, id_type
HAVING MEDIAN(card_price) >= 2000000
"""


class ExpensiveCarCustomers(RegularSegmentBuilder):
    name_segment_dict = {
        'expensive_car_customers_1mln': (547, 1954),
        'expensive_car_customers_2mln': (547, 1955),
    }

    number_of_days = 35

    def requires(self):
        return {
            'AutoFrontLog': LogProcessor(
                ProcessedAutoFrontLogForExpensiveCarCustomers,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            segment_query.format(
                input_table=inputs['AutoFrontLog'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
