#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.luigi_utils import OldNodesByNameCleaner
from crypta.profile.runners.segments.lib.coded_segments.travellers import TravellersWeekly

day_processor_query_template = u"""
$region_visits = (
    SELECT
        id,
        id_type,
        region as region_id,
        days as `date`
    FROM CONCAT({geo_table})
    FLATTEN LIST BY days
);

$capitals_and_regions = (
    SELECT
        id,
        id_type,
        `date`,
        Geo::IsRegionInRegion(region_id, 213) AS is_moscow,
        NOT Geo::IsRegionInRegion(region_id, 213) AND Geo::IsRegionInRegion(region_id, 1) AS is_moscow_oblast,
        Geo::IsRegionInRegion(region_id, 2) AS is_piter,
        NOT Geo::IsRegionInRegion(region_id, 2) AND Geo::IsRegionInRegion(region_id, 10174) AS is_lenin_oblast,
        Geo::IsRegionInRegion(region_id, 54) AS is_ekat,
        NOT Geo::IsRegionInRegion(region_id, 54) AND Geo::IsRegionInRegion(region_id, 11162) AS is_ekat_oblast,
        Geo::IsRegionInRegion(region_id, 43) AS is_kazan,
        NOT Geo::IsRegionInRegion(region_id, 43) AND Geo::IsRegionInRegion(region_id, 11119) AS is_tatarstan,
        Geo::IsRegionInRegion(region_id, 65) AS is_novosib,
        NOT Geo::IsRegionInRegion(region_id, 43) AND Geo::IsRegionInRegion(region_id, 11316) AS is_novosib_oblst,
    FROM $region_visits
);

$travel_days = (
    SELECT
        id,
        id_type,
        `date`
    FROM (
        SELECT
            id,
            id_type,
            `date`,
            CASE
                WHEN MAX(is_moscow) AND MAX(is_moscow_oblast) THEN True
                WHEN MAX(is_piter) AND MAX(is_lenin_oblast) THEN True
                WHEN MAX(is_ekat) AND MAX(is_ekat_oblast) THEN True
                WHEN MAX(is_kazan) AND MAX(is_tatarstan) THEN True
                WHEN MAX(is_novosib) AND MAX(is_novosib_oblst) THEN True
                ELSE False
            END AS travel_day,
        FROM $capitals_and_regions
        GROUP BY id, id_type, `date`
    )
    WHERE travel_day
);

$visits_count = (
    SELECT
        id,
        id_type,
        COUNT_IF('{start_date_this_week}' <= `date` AND '{end_date_this_week}' >= `date`) AS this_week,
        COUNT_IF('{start_date_previous_week}' <= `date` AND '{end_date_previous_week}' >= `date`) AS previous_week
    FROM $travel_days
    GROUP BY id, id_type
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    'daily_travellers' AS segment_name
FROM $visits_count
WHERE this_week >= 2 AND previous_week >= 2;
"""


class DailyTravellers(RegularSegmentBuilder):
    keyword = 557

    name_segment_dict = {
        'daily_travellers': 18423940,
    }

    def requires(self):
        dates = sorted(self.yt.list(config.WEEKLY_GEO_FOLDER))[-2:]

        tasks_to_run = []
        for date in dates:
            tasks_to_run.append(TravellersWeekly(date))

        return {
            'DayProcessorTasks': tasks_to_run,
            'CleanOldDayProcessors': OldNodesByNameCleaner(
                date=self.date,
                folder=tasks_to_run[0].output_directory,
                lifetime=372,
                date_format='%Y-%m-%d',
            ),
        }

    def build_segment(self, inputs, output_path):
        last_processed = self.yt.TablePath(
            os.path.join(config.WEEKLY_GEO_FOLDER, self.yt.list(config.WEEKLY_GEO_FOLDER)[-1]))

        if self.yt.get_attribute(last_processed, 'modification_time').split('T')[0] != self.date:
            self.yt.set_attribute(
                output_path,
                'generate_date',
                self.date,
            )
            return

        input_tables = []
        for target in self.input()['DayProcessorTasks']:
            input_tables.append("`{}`".format(target.table))

        self.yql.query(
            day_processor_query_template.format(
                geo_table=', '.join(input_tables),
                output_table=output_path,
                date=self.date,
                start_date_this_week=date_helpers.get_date_from_past(self.date, 6),
                end_date_this_week=self.date,
                start_date_previous_week=date_helpers.get_date_from_past(self.date, 13),
                end_date_previous_week=date_helpers.get_date_from_past(self.date, 7),
            ),
            transaction=self.transaction,
        )
