#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

import luigi

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.clients.clickhouse import ClickhouseClient
from crypta.profile.utils.luigi_utils import ExternalInput, BaseYtTask, YtDailyRewritableTarget
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder


metrika_users_day_query = """
SELECT DISTINCT PassportUserID, toString(EventDate)
FROM hits_all
WHERE CounterID == 24226447
    AND EventDate > toDate('{last_date}')
    AND EventDate <= toDate('{today_date}')
    AND PassportUserID != 0
"""

app_metrika_users_day_query = """
SELECT DISTINCT UUID, toString(EventDate)
FROM mobgiga.generic_events_all
WHERE EventDate > toDate('{last_date}')
    AND EventDate <= toDate('{today_date}')
    AND (APIKey == 137715)
    AND (SessionType == 0 OR EventType = 1)"""

drop_old_records_query = """
INSERT INTO `{storage_table}` WITH TRUNCATE
SELECT id, id_type, `date`
FROM `{storage_table}`
WHERE `date` > '{year_ago_date}'
"""

METRIKA_USERS_STORAGE_TABLE = os.path.join(config.LONG_TERM_DATA_FOLDER, 'metrika_users')


class UpdateMetrikaUsersStorage(BaseYtTask):
    date = luigi.Parameter()

    def requires(self):
        return ExternalInput(METRIKA_USERS_STORAGE_TABLE)

    def output(self):
        return YtDailyRewritableTarget(METRIKA_USERS_STORAGE_TABLE, self.date)

    def run(self):
        last_date = self.yt.get_attribute(METRIKA_USERS_STORAGE_TABLE, 'generate_date')

        clickhouse = ClickhouseClient(logger=self.logger)
        result = clickhouse.make_query(
            metrika_users_day_query.format(
                last_date=last_date,
                today_date=self.date,
            ),
        )

        entries = clickhouse.result_to_dict_list(result, ['id', 'date'])

        result = clickhouse.make_query(
            app_metrika_users_day_query.format(
                last_date=last_date,
                today_date=self.date,
            ),
        )

        app_entries = clickhouse.result_to_dict_list(result, ['id', 'date'])

        with self.yt.Transaction() as transaction:
            self.yt.write_table(
                self.yt.TablePath(self.output().table, append=True),
                [{'id': str(entry['id']), 'id_type': 'puid', 'date': entry['date']} for entry in entries],
            )

            self.yt.write_table(
                self.yt.TablePath(self.output().table, append=True),
                [{'id': str(entry['id']), 'id_type': 'uuid', 'date': entry['date']} for entry in app_entries],
            )

            self.yql.query(
                drop_old_records_query.format(
                    storage_table=METRIKA_USERS_STORAGE_TABLE,
                    year_ago_date=date_helpers.get_date_from_past(self.date, 365),
                ),
                transaction=transaction,
            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )


segment_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, 'metrika_users' AS segment_name
FROM `{input_table}`
GROUP BY id, id_type
"""


class MetrikaUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'metrika_users': (549, 2085),
    }

    def requires(self):
        return UpdateMetrikaUsersStorage(self.date)

    def build_segment(self, inputs, output_path):
        self.yql.query(
            segment_query.format(
                input_table=METRIKA_USERS_STORAGE_TABLE,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
