#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import (
    LaLParams,
    RegularSegmentBuilder,
)

plus_subscribers = u"""
$plus_subscribers = (
    SELECT
        puid AS id,
        'puid' AS id_type,
        'plus_subscribers' AS segment_name
    FROM (
        SELECT DISTINCT puid
        FROM `{input_table}`
        WHERE state == 'active' AND plus IS NOT NULL AND plus
    )
);

$active_subscriptions = (
    SELECT
        puid AS id,
        'puid' AS id_type,
        segment_name
    FROM (
        SELECT
            DISTINCT puid,
            CASE
                WHEN bundle == 'kp-amediateka' THEN 'kp-amediateka'
                WHEN bundle == 'new-plus' AND plan IN AsSet('child', 'parent') THEN 'plus-multi'
                WHEN bundle == 'new-plus' AND plan == 'single' THEN 'basic-plus'
                ELSE NULL
            END AS segment_name
        FROM `{input_table}`
        WHERE state == 'active'
    )
    WHERE segment_name IS NOT NULL
);

$former_subscribers = (
    SELECT *
    FROM (
        SELECT
            puid AS id,
            'puid' AS id_type,
            CASE
                WHEN bundle == 'kp-amediateka' THEN 'former-kp-amediateka'
                WHEN bundle == 'new-plus' AND plan IN AsSet('child', 'parent') THEN 'former-plus-multi'
                WHEN bundle == 'new-plus' AND plan == 'single' THEN 'former-basic-plus'
                ELSE NULL
            END AS segment_name
        FROM (
            SELECT
                puid,
                bundle,
                plan
            FROM `{input_table}`
            GROUP BY puid, bundle, plan
            HAVING AGGREGATE_LIST_DISTINCT(state) == AsList('churned') AND MAX(end_date) < '{last_date}'
        )
    )
    WHERE segment_name IS NOT NULL
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT * FROM $plus_subscribers
UNION ALL
SELECT * FROM $active_subscriptions
UNION ALL
SELECT * FROM $former_subscribers;

INSERT INTO `{plus_sample_table}` WITH TRUNCATE
SELECT
    CAST(yandexuid AS String) AS yandexuid,
    'plus_subscribers' AS segment_name
FROM (
    SELECT matching.yandexuid AS yandexuid
    FROM $plus_subscribers AS segment
    INNER JOIN `{indevice_yandexuid_matching}` AS matching
    USING (id, id_type)
    WHERE segment.segment_name == 'plus_subscribers'
)
GROUP BY yandexuid
"""


class PlusSubscribers(RegularSegmentBuilder):
    name_segment_dict = {
        'plus_subscribers': (549, 1917),
        'kp-amediateka': (549, 1161),
        'plus-multi': (549, 2105),
        'basic-plus': (549, 2104),
        'former-kp-amediateka': (549, 1195),
        'former-plus-multi': (549, 2103),
        'former-basic-plus': (549, 2102),
    }

    def requires(self):
        return ExternalInput(os.path.join(config.SUBSCRIBERS_CUBE, self.date))

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as plus_sample_table:
            self.yql.query(
                query_string=plus_subscribers.format(
                    input_table=inputs.table,
                    output_table=output_path,
                    indevice_yandexuid_matching=config.INDEVICE_YANDEXUID,
                    plus_sample_table=plus_sample_table,
                    date=self.date,
                    last_date=date_helpers.get_date_from_past(self.date, 3),
                    logger=self.logger,
                ),
                transaction=self.transaction,
            )

            self.prepare_samples_for_lal(
                plus_sample_table,
                id_field='yandexuid',
                lals_params=[LaLParams(
                    name='plus_subscribers',
                    id=1393,
                    type='lal_internal',
                    coverage=10000000,
                    include_input=False,
                )],
            )
