#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


yql_query_template = u"""
$hair_vendors = AsSet(
    206928, --Dyson
    152872, --Rowenta
    152870, --Philips
    434163, --Babyliss
    624024, --Remington
    242456, --Bork
    152903, --Braun
    288426, --Polaris
    2310287, --Redmond
    153082, --Panasonic
    );

$vacuum_cleaners_vendors = AsSet(
    206928, --Dyson
    153061, --Samsung
    152849, --Tefal
    152900, --Bosch
    152870, --Philips
    152898, --Electrolux
    7701962, --Xiaomi
    153074, --LG
    242456, --Bork
    7695403, --Kitfort
    11042954, --Dexp
    2310287, --Redmond
    152970, --Gorenje
    239137, --Hoover
    152872, --Rowenta
    2410082, --Bissell
);

$climat_vendors = AsSet(
    242456, --Bork
    152898, --Electrolux
    288863, --Venta
    590377, --Boneco
    152870, --Philips
    153074, --LG
    7701962, --Xiaomi
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    segment_name
FROM (
    SELECT
        CASE
            WHEN passport_uid != 0 THEN CAST(passport_uid AS String)
            ELSE yandex_uid
        END AS id,
        CASE
            WHEN passport_uid != 0 THEN 'puid'
            ELSE 'yandexuid'
        END AS id_type,
        CASE
            WHEN hyper_id == 521440226 AND String::ToLower(original_query) LIKE '%мини%' AND String::ToLower(original_query) LIKE '%станция%' THEN 'mini_station'
            WHEN hyper_id == 1971204201 AND String::ToLower(original_query) LIKE '%станция%' THEN 'big_station'
            WHEN hyper_cat_id IN AsSet(16336734, 16336768, 16336769) AND vendor_id IN $hair_vendors THEN 'hair'
            WHEN hyper_cat_id IN AsSet(16302535, 16302537, 16302536, 278341) AND vendor_id IN $vacuum_cleaners_vendors THEN 'vacuum'
            WHEN hyper_cat_id == 734595 AND vendor_id IN $climat_vendors THEN 'climat'
            ELSE NULL
        END AS segment_name
    FROM `{input_table}`
    WHERE
        pp IN AsSet(7, 48, 1007, 1607, 1707, 1807)
        AND (passport_uid != 0 OR yandex_uid IS NOT NULL)
)
GROUP BY id, id_type, segment_name
HAVING segment_name IS NOT NULL;
"""


class ProcessMarketShowsLogForSearchedOnMarket(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.MARKET_SHOW_LOG_FOLDER,
                self.date,
            ),
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            yql_query_template.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class SearchedOnMarket(RegularSegmentBuilder):
    name_segment_dict = {
        'big_station': (549, 2082),
        'mini_station': (549, 2081),
        'hair': (557, 17937562),
        'vacuum': (557, 17937556),
        'climat': (557, 17937559),

    }

    number_of_days = 35

    def requires(self):
        return {
            'ProcessedMarketShowsLogForSearchedOnMarket': LogProcessor(
                ProcessMarketShowsLogForSearchedOnMarket,
                self.date,
                self.number_of_days,
            )
        }

    def build_segment(self, inputs, output_path):
        self.yt.unique(
            inputs['ProcessedMarketShowsLogForSearchedOnMarket'].table,
            output_path,
            unique_by=['id', 'id_type', 'segment_name'],
        )
