#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
from urlparse import urljoin

import requests

from crypta.profile.utils.config import config
from crypta.profile.utils.config.secrets import get_secrets
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.clients.clickhouse import ClickhouseClient
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


yandex_staff_clickhouse_query = """
SELECT DISTINCT(UserID)
FROM hits_all
WHERE CounterID IN (156784, 153432, 1004067)
    AND CounterClass == 0
    AND EventDate == toDate('{date}')
"""

STAFF_API_URL = 'http://staff-api.yandex-team.ru'


class ProcessedBsWatchLogForYandexStaff(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.BS_WATCH_LOG_FOLDER,
                self.date,
            ),
        )

    def process_day(self, inputs, output_path):
        clickhouse = ClickhouseClient(logger=self.logger)
        result = clickhouse.make_query(
            yandex_staff_clickhouse_query.format(date=self.date)
        )
        if not result:
            self.logger.error('Empty result')
        entries = clickhouse.result_to_dict_list(result, ['id'])

        self.yt.create_empty_table(
            output_path,
            schema={
                'id': 'string',
            },
        )
        self.yt.write_table(output_path, entries)


segment_query = """
PRAGMA yt.InferSchema;

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT *
FROM (
    SELECT id, 'yandexuid' AS id_type, 'yandex_staff' AS segment_name
    FROM `{input_table}`
    WHERE id IS NOT NULL AND id != '' AND id != '0'
)
UNION ALL
SELECT *
FROM (
    SELECT
        Identifiers::HashMd5Email(email) AS id,
        'email_md5' AS id_type,
        'yandex_staff' AS segment_name
    FROM `{staff_logins_table}`
)
"""


class YandexStaff(RegularSegmentBuilder):
    number_of_days = 35

    name_segment_dict = {
        'yandex_staff': (549, 1410),
    }

    def requires(self):
        return {
            'ProcessedMetrics': LogProcessor(
                ProcessedBsWatchLogForYandexStaff,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as staff_logins_table:
            response = requests.get(
                urljoin(STAFF_API_URL, 'v3/persons'),
                params={'official.is_dismissed': 'false', '_fields': 'yandex.login'},
                headers={'Authorization': 'OAuth {token}'.format(token=get_secrets().get_secret('ROBOT_UNICORN_STAFF_OAUTH'))},
            ).json()

            records = response['result']

            while response['page'] != response['pages']:
                response = requests.get(
                    response['links']['next'],
                    headers={'Authorization': 'OAuth {token}'.format(token=get_secrets().get_secret('ROBOT_UNICORN_STAFF_OAUTH'))},
                ).json()

                records.extend(response['result'])

            logins = []

            for record in records:
                if record['yandex']['login']:
                    logins.append({'email': record['yandex']['login'] + '@yandex.ru'})

            self.yt.write_table(
                staff_logins_table,
                logins,
            )

            self.yql.query(
                segment_query.format(
                    input_table=inputs['ProcessedMetrics'].table,
                    output_table=output_path,
                    staff_logins_table=staff_logins_table,
                ),
                transaction=self.transaction,
                udf_resource_dict={'libcrypta_identifier_udf.so': config.CRYPTA_IDENTIFIERS_UDF_RESOURCE},
            )
