#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


kbt_customers_query = """
$other_categories = AsList(
    43, 53, 183, 184, 186, 187, 191, 195,
    201, 221, 222, 224, 227, 229, 235, 236,
    238, 239, 240, 241, 244, 249, 255, 1197,
    1541, 1543, 1544, 1620, 1727, 10013, 10021,
    10023, 18305, 10174, 10239, 10279, 10286,
    10289, 10407, 20406, 21466, 23587
);

$fridge_category = 254;
$laptop_category = 55;
$phone_category = 444;
$tablet_category = 19205;
$tvs_category = 1145;

$all_categories = ListExtend(
    $other_categories,
    AsList(
        $fridge_category, $laptop_category, $phone_category,
        $tablet_category, $tvs_category
    )
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, segment_name
FROM (
    SELECT
        yandex_uid AS id,
        'yandexuid' AS id_type,
        CASE
            WHEN category_id IN $other_categories THEN 'kbt_all_customers'
            WHEN category_id == $fridge_category THEN 'kbt_fridges_customers'
            WHEN category_id == $laptop_category THEN 'kbt_laptops_customers'
            WHEN category_id == $phone_category THEN 'kbt_phones_customers'
            WHEN category_id == $tablet_category THEN 'kbt_tablets_customers'
            WHEN category_id == $tvs_category THEN 'kbt_tvs_customers'
            ELSE ''
        END AS segment_name
    FROM `{input_table}`
    WHERE yandex_uid IS NOT NULL AND price >= 10000 AND price <= 60000 AND category_id IN $all_categories
)
GROUP BY id, id_type, segment_name
"""


class MarketShowsLogForKbtCustomers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.MARKET_SHOW_LOG_FOLDER, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            kbt_customers_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class KbtCustomers(RegularSegmentBuilder):
    name_segment_dict = {
        'kbt_all_customers': 529,
        'kbt_fridges_customers': 530,
        'kbt_laptops_customers': 531,
        'kbt_phones_customers': 532,
        'kbt_tablets_customers': 533,
        'kbt_tvs_customers': 534,
    }

    keyword = 216
    number_of_days = 7

    def requires(self):
        return {
            'MarketShowsLog': LogProcessor(
                MarketShowsLogForKbtCustomers,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yt.unique(
            [inputs['MarketShowsLog'].table],
            output_path,
            unique_by=['id', 'id_type', 'segment_name'],
        )
