#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.profile.runners.segments.lib.coded_segments.common import deep_visits
from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor
from crypta.profile.utils.utils import get_matching_table

kfc_visitors_query_template = """
$apps = AsSet(
    'ru.foodfox.courier.debug.releaseserver',
    'com.yandex.courier',
    'com.deliveryclub.rider',
);

$permalinks = (
    SELECT DISTINCT(AGGREGATE_LIST(permalink))
    FROM `{organization_categories}`
    WHERE name_ru == 'KFC'
);

$couriers = (
    SELECT id, id_type
    FROM `{app_metrica}`
    WHERE app in $apps
    GROUP BY id, id_type
);

$visited_no_couriers = (
    SELECT visits.devid as id
    FROM `{deep_visits}` as visits
    LEFT ONLY JOIN $couriers as couriers
    ON visits.devid == couriers.id
    WHERE ListHas($permalinks, permalink)
    GROUP BY visits.devid
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT visits.id as id,
       'idfa' as id_type
FROM $visited_no_couriers as visits
LEFT SEMI JOIN `{matching_idfa}` as matching
USING(id)

UNION ALL

SELECT visits.id as id,
       'gaid' as id_type
FROM $visited_no_couriers as visits
LEFT SEMI JOIN `{matching_gaid}` as matching
USING(id);
"""


def map_kfc_visitors(row):
    yield row


def reduce_kfc_visitors(key, rows):
    data = dict(key)
    data['segment_name'] = 'kfc_visitors'
    yield data


class ProcessedDeepVisitLogForKfcVisitors(DayProcessor):
    def requires(self):
        return {
            'AppMetrica': ExternalInput(config.DEVID_BY_APP_MONTHLY_TABLE),
            'DeepVisits': deep_visits.org_visits_deep_external_input(self.date),
        }

    def process_day(self, inputs, output_path):

        self.yql.query(
            kfc_visitors_query_template.format(
                organization_categories=config.ORGANIZATION_CATEGORIES,
                deep_visits=inputs['DeepVisits'].table,
                app_metrica=inputs['AppMetrica'].table,
                matching_idfa=get_matching_table('idfa', 'crypta_id'),
                matching_gaid=get_matching_table('gaid', 'crypta_id'),
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class KfcVisitors(RegularSegmentBuilder):
    keyword = 557
    number_of_days = 182

    name_segment_dict = {
        'kfc_visitors': 16113988,
    }

    def requires(self):
        return LogProcessor(ProcessedDeepVisitLogForKfcVisitors, self.date, self.number_of_days)

    def build_segment(self, inputs, output_path):

        self.yt.run_map_reduce(
            map_kfc_visitors,
            reduce_kfc_visitors,
            inputs.table,
            output_path,
            reduce_by=['id', 'id_type']
        )
