#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


day_processor_query = u"""
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    `UUID` AS id,
    CAST(CAST(DateTime::FromSeconds(CAST(StartDate AS Uint32)) AS Date) AS String) AS StartDate,
    COUNT(*) AS event_count
FROM `{input_table}`
WHERE APIKey == 10321
    AND EventName = 'APP_LAUNCH_STATUS_EVENT'
    AND Manufacturer IS NOT NULL
    AND Manufacturer IN ('Samsung', 'Xiaomi')
    AND `UUID` IS NOT NULL
GROUP BY `UUID`, StartDate
"""


class MetrikaMobileLogForLoyalToLauncherInstall(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(
                config.APPMETRICA_YANDEX_EVENTS_LOG_FOLDER,
                self.date,
            ),
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            query_string=day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = """
$to_date = ($f) -> {{
    RETURN CAST($f AS Date)
}};

$get_interval_days = ($d1, $d2) -> {{
    RETURN DateTime::ToDays($to_date($d1) - $to_date($d2))
}};

$min_sessions_per_day = 3;

$end_date = '{end_date}';
$start_date = CAST((CAST($end_date AS Date) - DateTime::IntervalFromDays({number_of_days})) AS String);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    'uuid' AS id_type,
    'launcher_install' AS segment_name
FROM `{input_table}`
WHERE StartDate >= $start_date AND StartDate <= $end_date
GROUP BY id
HAVING SUM(event_count) > $get_interval_days(MAX(StartDate), MIN(StartDate)) * $min_sessions_per_day
"""


class LoyalToLauncherInstall(RegularSegmentBuilder):
    name_segment_dict = {
        'launcher_install': 1776,
    }

    keyword = 549
    number_of_days = 7

    def requires(self):
        return LogProcessor(
            MetrikaMobileLogForLoyalToLauncherInstall,
            self.date,
            self.number_of_days,
        )

    def build_segment(self, inputs, output_path):
        self.yql.query(
            query_string=segment_query.format(
                input_table=inputs.table,
                output_table=output_path,
                end_date=self.date,
                number_of_days=self.number_of_days,
            ),
            transaction=self.transaction,
        )
