#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import tempfile

from crypta.lib.python.custom_ml.tools.features import get_features_dict_unraveled
from crypta.lib.python.custom_ml.training_config import segment_feature_types
from crypta.profile.utils import utils
from crypta.profile.utils.config import config
from crypta.profile.utils.yql_utils import Yql

resources = {
    # classification
    'bed': 'CRYPTA_BED_SEGMENTS_CATBOOST_MODEL',
    'cash_settlement_services': 'CRYPTA_CASH_SETTLEMENT_SERVICES_SEGMENTS_CATBOOST_MODEL',
    'cosmetics': 'CRYPTA_COSMETICS_SEGMENTS_CATBOOST_MODEL',
    'credit_card': 'CRYPTA_CREDIT_CARD_SEGMENTS_CATBOOST_MODEL',
    'custom_income': 'CRYPTA_CUSTOM_INCOME_SEGMENTS_CATBOOST_MODEL',
    'dating': 'CRYPTA_DATING_SEGMENTS_CATBOOST_MODEL',
    'debit_cards_activation': 'CRYPTA_DEBIT_CARDS_ACTIVATION_SEGMENTS_CATBOOST_MODEL',
    'ebook': 'CRYPTA_EBOOK_SEGMENTS_CATBOOST_MODEL',
    'electronic': 'CRYPTA_ELECTRONIC_SEGMENTS_CATBOOST_MODEL',
    'english_online_education': 'CRYPTA_ENGLISH_ONLINE_EDUCATION_SEGMENTS_CATBOOST_MODEL',
    'food_delivery': 'CRYPTA_FOOD_DELIVERY_SEGMENTS_CATBOOST_MODEL',
    'furniture': 'CRYPTA_FURNITURE_SEGMENTS_CATBOOST_MODEL',
    'game_login': 'CRYPTA_GAME_SEGMENTS_CATBOOST_MODEL',
    'generic_car_credit_approval': 'CRYPTA_GENERIC_CAR_CREDIT_APPROVAL_SEGMENTS_CATBOOST_MODEL',
    'generic_credit_approval': 'CRYPTA_GENERIC_CREDIT_APPROVAL_SEGMENTS_CATBOOST_MODEL',
    'generic_insurance': 'CRYPTA_GENERIC_INSURANCE_SEGMENTS_CATBOOST_MODEL',
    'generic_microfinance_approval': 'CRYPTA_GENERIC_MICROFINANCE_APPROVAL_SEGMENTS_CATBOOST_MODEL',
    'generic_scoring': 'CRYPTA_GENERIC_SCORING_SEGMENTS_CATBOOST_MODEL',
    'grocery_delivery': 'CRYPTA_GROCERY_DELIVERY_SEGMENTS_CATBOOST_MODEL',
    'healthy_food': 'CRYPTA_HEALTHY_FOOD_SEGMENTS_CATBOOST_MODEL',
    'home_depot': 'CRYPTA_HOME_DEPOT_SEGMENTS_CATBOOST_MODEL',
    'hotel': 'CRYPTA_HOTEL_SEGMENTS_CATBOOST_MODEL',
    'investment': 'CRYPTA_INVESTMENT_SEGMENTS_CATBOOST_MODEL',
    'legal_entities': 'CRYPTA_LEGAL_ENTITIES_SEGMENTS_CATBOOST_MODEL',
    'legal_office_visits': 'CRYPTA_LEGAL_OFFICE_VISITS_SEGMENTS_CATBOOST_MODEL',
    'market': 'CRYPTA_MARKET_SEGMENTS_CATBOOST_MODEL',
    'market_rfm': 'CRYPTA_MARKET_RFM_SEGMENTS_CATBOOST_MODEL',
    'marriage': 'CRYPTA_MARRIAGE_SEGMENTS_CATBOOST_MODEL',
    'medical_clinics': 'CRYPTA_MEDICAL_CLINIC_SEGMENTS_CATBOOST_MODEL',
    'mortgage_approval': 'CRYPTA_MORTGAGE_APPROVAL_SEGMENTS_CATBOOST_MODEL',
    'online_cinema': 'CRYPTA_ONLINE_CINEMA_SEGMENTS_CATBOOST_MODEL',
    'online_payment': 'CRYPTA_ONLINE_PAYMENT_SEGMENTS_CATBOOST_MODEL',
    'online_sales_register': 'CRYPTA_ONLINE_SALES_REGISTER_SEGMENTS_CATBOOST_MODEL',
    'online_shopping': 'CRYPTA_ONLINE_SHOPPING_SEGMENTS_CATBOOST_MODEL',
    'pharmacy': 'CRYPTA_PHARMACY_SEGMENTS_CATBOOST_MODEL',
    'realty_visit': 'CRYPTA_REALTY_VISIT_SEGMENTS_CATBOOST_MODEL',
    'tv_viewers': 'CRYPTA_TV_VIEWERS_SEGMENTS_CATBOOST_MODEL',
    'windows_installation': 'CRYPTA_WINDOWS_INSTALLATION_SEGMENTS_CATBOOST_MODEL',

    # regression
    'heavy_suite': 'CRYPTA_HEAVY_SUITE_SEGMENTS_CATBOOST_MODEL',
}


def get_model_tag(model_name):
    resource_type = resources[model_name]
    return 'https://proxy.sandbox.yandex-team.ru/last/{}?attrs={{"released":"stable"}}'.format(resource_type)


def upload_models_and_features_mapping(logger, yt):
    for model_name, resource in resources.items():
        logger.info('Uploading {} model'.format(model_name))
        with tempfile.NamedTemporaryFile() as model_file:
            utils.download_file_from_sandbox(resource, 'prestable', '', model_file.name)

            model_file.seek(0)
            attributes = {'ttl': '90', 'released': 'stable'}
            utils.upload_to_sandbox(
                path=model_file.name,
                resource_type=resource,
                description='model for trainable segment',
                owner='CRYPTA',
                attributes=attributes,
                logger=logger,
            )

    cat_features_dict_unraveled, _ = get_features_dict_unraveled(
        yt=yt,
        segment_feature_types=segment_feature_types,
        vector_size=config.VECTOR_SIZE,
    )

    with tempfile.NamedTemporaryFile() as dict_file:
        json.dump(cat_features_dict_unraveled, dict_file)
        dict_file.seek(0)
        attributes = {'ttl': '90', 'released': 'stable'}
        utils.upload_to_sandbox(
            path=dict_file.name,
            resource_type='CRYPTA_FEATURES_MAPPING_FOR_TRAINABLE_SEGMENTS',
            description='features mapping for trainable segments',
            owner='CRYPTA',
            attributes=attributes,
            logger=logger,
        )


def nirvana_train_custom_segment_model(train_helper):
    model, metrics_to_send = train_helper.get_model_and_metrics()

    with tempfile.NamedTemporaryFile() as model_file:
        model.save_model(model_file.name)
        model_file.seek(0)
        attributes = {'ttl': '90', 'released': 'prestable'}
        utils.upload_to_sandbox(
            path=model_file.name,
            resource_type=train_helper.model_params.resource_type,
            description=train_helper.model_params.model_description_in_sandbox,
            owner='CRYPTA',
            attributes=attributes,
            logger=train_helper.logger,
        )

    utils.report_ml_metrics_to_solomon(
        service=config.SOLOMON_TRAINABLE_SEGMENTS_SERVICE,
        metrics_to_send=metrics_to_send,
    )

    train_helper.yt.set_attribute(
        train_helper.model_params.train_sample_path,
        'model_training_date',
        train_helper.date,
    )


def get_custom_segment_model_train_sample(train_helper, transaction=None):
    yql = Yql(train_helper.logger, train_helper.yt)

    yql.query(
        query_string=train_helper.model_params.make_train_sample_query,
        transaction=transaction,
    )

    train_helper.yt.set_attribute(
        train_helper.model_params.train_sample_path,
        'generate_date',
        train_helper.date,
    )
