#!/usr/bin/env python
# -*- coding: utf-8 -*-
import luigi

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput, YtDailyRewritableTarget, BaseYtTask
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder

GAMES_COUNT_LOWER_BOUND = 10
GAMES_COUNT_UPPER_BOUND = 200


class GamersMapper(object):
    def __init__(self, apps):
        self.game_apps = apps

    def __call__(self, row):
        installed_apps = set(app.lower() for app in row['apps'])
        games_count = len(installed_apps.intersection(self.game_apps))
        if GAMES_COUNT_LOWER_BOUND <= games_count <= GAMES_COUNT_UPPER_BOUND:
            yield {
                'id': row['id'],
                'id_type': row['id_type'],
                'segment_name': 'segment-mobile_games',
            }


class GetMobileGamers(BaseYtTask):
    date = luigi.Parameter()
    task_group = 'coded_segments'

    def requires(self):
        return ExternalInput(config.APP_METRICA_MONTH)

    def output(self):
        return YtDailyRewritableTarget(
            config.MOBILE_GAMERS_DEVIDS,
            self.date,
        )

    def run(self):
        with self.yt.Transaction():
            game_apps = set()
            for row in self.yt.read_table(config.MOBILE_APP_CATEGORIES):
                for category in row['raw_categories']:
                    if 'game' in category:
                        game_apps.add(row['bundleId'].lower())
                        break

            self.yt.create_empty_table(
                self.output().table,
                schema={
                    'id': 'string',
                    'id_type': 'string',
                    'segment_name': 'string',
                },
            )

            self.yt.run_map(
                GamersMapper(game_apps),
                self.yt.TablePath(self.input().table, columns=('id', 'id_type', 'apps')),
                self.output().table,
                spec={
                    'mapper': {'memory_limit': 2 * 1024 * 1024 * 1024},
                },

            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )


class MobileGamers(RegularSegmentBuilder):
    name_segment_dict = {
        'segment-mobile_games': (547, 1905),
    }

    def requires(self):
        return GetMobileGamers(self.date)

    def build_segment(self, inputs, output_path):
        self.yt.copy(
            inputs.table,
            output_path,
            force=True,
        )
