#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor

from crypta.profile.runners.segments.lib.coded_segments.mobile_operators_users_by_prefix import MobileOperatorsUsersByPrefix

code_to_name = {
    '(250, 1)': 'MTS',
    '(250, 2)': 'Megafon',
    '(250, 20)': 'Tele2',
    '(250, 99)': 'Beeline',
    '(434, 5)': 'Ucell',
    '(401, 77)': 'Tele2Kz',
}

process_day_query = """
$operators = ToDict(AsList({operators}));

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    $operators[simcards_operators.0] as segment_name,
    id,
    id_type,
FROM `{input_table}` as mobile_metrics
FLATTEN DICT BY simcards_operators
WHERE simcards_operators.0 IN DictKeys($operators);
"""

mobile_operator_query = """
$found_in_logs = (
    SELECT
        matching.cryptaId as cryptaId,
        operators.segment_name as segment_name,
        operators.id as id,
        operators.id_type as id_type
    FROM `{input_table}` as operators
    INNER JOIN `{vertices_no_multi_profile_table}` AS matching
    USING(id, id_type)
);

INSERT INTO `{output_table}` WITH TRUNCATE

SELECT
    by_prefix.segment_name as segment_name,
    by_prefix.id as id,
    by_prefix.id_type as id_type
FROM (
    SELECT
        operators.segment_name as segment_name,
        matching.cryptaId as cryptaId,
        operators.id as id,
        operators.id_type as id_type,
    FROM `{operator_by_prefix}` as operators
    INNER JOIN `{vertices_no_multi_profile_table}` as matching
    USING(id, id_type)
    WHERE operators.segment_name != 'Virtual'
) as by_prefix
LEFT ONLY JOIN $found_in_logs as logs
USING(cryptaId)
GROUP BY by_prefix.segment_name, by_prefix.id, by_prefix.id_type

UNION ALL

SELECT segment_name, id, id_type
FROM $found_in_logs
GROUP BY segment_name, id, id_type;
"""


class ProcessedMetricsForMobileOperatorsUsers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.MOBILE_DEV_INFO.format(self.date)))

    def process_day(self, inputs, output_path):
        self.yql.query(
            query_string=process_day_query.format(
                input_table=inputs.table,
                output_table=output_path,
                operators=', '.join(['({}, "{}")'.format(key, name) for key, name in code_to_name.iteritems()]),
            ),
            transaction=self.transaction,
        )


class MobileOperatorsUsers(RegularSegmentBuilder):
    name_segment_dict = {
        'Megafon': (547, 1960),
        'MTS': (547, 1959),
        'Beeline': (547, 1958),
        'Tele2': (547, 1957),
        'Ucell': (547, 1793),
        'Tele2Kz': (557, 17524351),
    }
    number_of_days = 28

    def requires(self):
        return {
            'ProcessedMetrics': LogProcessor(
                ProcessedMetricsForMobileOperatorsUsers,
                date_helpers.get_yesterday(self.date),
                self.number_of_days,
            ),
            'Prefix': MobileOperatorsUsersByPrefix(self.date),
            'OperatorByPrefix': ExternalInput(
                os.path.join(
                    config.SEGMENT_RAW_OUTPUT_FOLDER,
                    MobileOperatorsUsersByPrefix(self.date).__class__.__name__,
                )
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            mobile_operator_query.format(
                input_table=inputs['ProcessedMetrics'].table,
                vertices_no_multi_profile_table=config.VERTICES_NO_MULTI_PROFILE,
                operator_by_prefix=inputs['OperatorByPrefix'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
