#!/usr/bin/env python
# -*- coding: utf-8 -*-

from datetime import timedelta, datetime
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.utils import get_matching_table


multidevice_puid_query_template = """
$idfa_gaid = (
SELECT * FROM `{gaid}`
WHERE date_end >= '{date}'
UNION ALL
SELECT * FROM `{idfa}`
WHERE date_end >= '{date}'
);

$all_devices = (SELECT
    puid_yuid.id as id,
    puid_yuid.id_type as id_type,
    puid_yuid.target_id as target_id,
    puid_yuid.target_id_type as target_id_type,
        device_type.ua as ua,
    IF(String::SplitToList(device_type.ua_profile, "|")[1] == 'desk',
       String::SplitToList(device_type.ua_profile, "|")[2]) as DeviceName,
    IF(String::SplitToList(device_type.ua_profile, "|")[1] == 'desk',
       String::SplitToList(device_type.ua_profile, "|")[3]) as DeviceModel,
FROM `{yandexuid}` as puid_yuid
INNER JOIN `{yuid_with_all_info}` as device_type
ON device_type.id == puid_yuid.target_id AND puid_yuid.target_id_type == device_type.id_type
WHERE device_type.ua is not null AND puid_yuid.date_end >= '{date}'

UNION ALL

SELECT
    mobile.id as id,
    mobile.id_type as id_type,
    mobile.target_id as target_id,
    mobile.target_id_type as target_id_type,
    String::SplitToList(device_type.ua_profile, "|")[2] as DeviceName,
    device_type.model as DeviceModel,
FROM $idfa_gaid as mobile
INNER JOIN `{app_metrica_month}` as device_type
ON mobile.target_id == device_type.id);


$devices = (
SELECT id, device
    FROM $all_devices
    WHERE DeviceName is not null OR
          DeviceModel is not null
GROUP BY id, AsList(DeviceName, DeviceModel) as device);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id as id,
    'puid' as id_type,
    'multidevice' as segment_name
FROM(
    SELECT id, COUNT(device) as device_count
    FROM $devices
    GROUP BY id)
WHERE device_count >= 4;
"""


class MultidevicePuid(RegularSegmentBuilder):

    name_segment_dict = {
        'multidevice': 1396,
    }

    keyword = 549
    number_of_days = 35

    def requires(self):
        return {'app_metrica_month': ExternalInput(config.APP_METRICA_MONTH),
                'yuid_with_all_info':  ExternalInput(config.YUID_WITH_ALL_INFO_TABLE),
                'idfa': ExternalInput(get_matching_table('puid', 'idfa', indevice=True)),
                'gaid': ExternalInput(get_matching_table('puid', 'gaid', indevice=True)),
                'yandexuid': ExternalInput(get_matching_table('puid', 'yandexuid', indevice=True))}

    def build_segment(self, inputs, output_path):
        self.yql.query(
            query_string=multidevice_puid_query_template.format(
                yuid_with_all_info=inputs['yuid_with_all_info'].table,
                app_metrica_month=inputs['app_metrica_month'].table,
                yandexuid=inputs['yandexuid'].table,
                idfa=inputs['idfa'].table,
                gaid=inputs['gaid'].table,
                date=datetime.strptime(self.date, "%Y-%m-%d").date() - timedelta(days=self.number_of_days),
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
