#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import (
    LaLParams,
    RegularSegmentBuilder,
)
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


MUSIC_GENRES_MAPPING = {
    'FOLK':         'music_folk_listeners',
    'RUSFOLK':      'music_folk_listeners',
    'EUROFOLK':     'music_folk_listeners',
    'AMERFOLK':     'music_folk_listeners',

    'JAZZ':         'music_jazz_listeners',
    'CONJAZZ':      'music_jazz_listeners',
    'TRADJAZZ':     'music_jazz_listeners',

    'ROCK':         'music_rock_listeners',
    'RUSROCK':      'music_rock_listeners',
    'POSTROCK':     'music_rock_listeners',
    'UKRROCK':      'music_rock_listeners',
    'RNR':          'music_rock_n_roll_listeners',

    'CLASSICAL':    'music_classical_listeners',

    'RAP':          'music_rap_listeners',
    'RUSRAP':       'music_rus_rap_listeners',

    'ELECTRONICS':  'music_electronic_listeners',

    'DANCE':        'music_dance_listeners',
    'HOUSE':        'music_dance_listeners',
    'DNB':          'music_dance_listeners',
    'TECHNO':       'music_dance_listeners',
    'TRANCE':       'music_dance_listeners',

    'RNB':          'music_rnb_listeners',
    'FUNK':         'music_rnb_listeners',

    'ALTERNATIVE':  'music_alternative_listeners',

    'BLUES':        'music_blues_listeners',

    'ESTRADA':      'music_estrada_listeners',
    'RUSESTRADA':   'music_estrada_listeners',

    'POP':          'music_pop_listeners',
    'DISCO':        'music_pop_listeners',
    'RUSPOP':       'music_pop_listeners',
    'ASIAPOP':      'music_pop_listeners',

    'METAL':        'music_metal_listeners',
    'CLASSICMETAL': 'music_metal_listeners',
    'EXPERIMENTAL': 'music_metal_listeners',
    'PROGMETAL':    'music_metal_listeners',
    'EPICMETAL':    'music_metal_listeners',
    'FOLKMETAL':    'music_metal_listeners',
    'NUMETAL':      'music_metal_listeners',

    'REGGAE':       'music_reggae_listeners',

    'INDIE':        'music_indie_listeners',
    'LOCAL-INDIE':  'music_indie_listeners',

    'PUNK':         'music_punk_listeners',
    'SKA':          'music_punk_listeners',
    'HARDCORE':     'music_punk_listeners',

    'BARD':         'music_bard_listeners',
    'RUSBARDS':     'music_bard_listeners',
    'ROMANCES':     'music_bard_listeners',

    'SOUL':         'music_light_listeners',
    'MEDITATION':   'music_light_listeners',
    'LOUNGE':       'music_light_listeners',

    'SHANSON':      'music_shanson_listeners',
}


def music_log_mapper(row):
    if 3.0 <= row['play_time'] <= 1000.0 and row['genre']:
        genres = set(row['genre']).intersection(MUSIC_GENRES_MAPPING.keys())

        if row['puid']:
            for genre in genres:
                yield {
                    'id': row['puid'],
                    'id_type': 'puid',
                    'segment_name': MUSIC_GENRES_MAPPING[genre],
                }
        if row['yandexuid']:
            for genre in genres:
                yield {
                    'id': row['yandexuid'],
                    'id_type': 'yandexuid',
                    'segment_name': MUSIC_GENRES_MAPPING[genre],
                }
        if row['uuid']:
            for genre in genres:
                yield {
                    'id': row['uuid'],
                    'id_type': 'uuid',
                    'segment_name': MUSIC_GENRES_MAPPING[genre],
                }


def unique_reducer(key, rows):
    yield key


class ProcessedMusicStreamsLogForGenresListeners(DayProcessor):
    def requires(self):
        return ExternalInput(
            os.path.join(config.MUSIC_STREAMS_LOG_FOLDER, self.date),
            columns=('puid', 'yandexuid', 'uuid', 'play_time', 'genre'),
        )

    def process_day(self, inputs, output_path):
        self.yt.run_map_reduce(
            music_log_mapper,
            unique_reducer,
            inputs.table,
            output_path,
            reduce_by=['id', 'id_type', 'segment_name'],
        )


join_query = """
PRAGMA yt.InferSchema;

$segment_ids = (
    SELECT id, id_type, segment_name
    FROM `{input_table}`
    GROUP BY id, id_type, segment_name
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT *
FROM $segment_ids;


INSERT INTO `{lal_sample_table}` WITH TRUNCATE
SELECT yandexuid, segment_name
FROM (
    SELECT COALESCE(yandexuid, CAST(id AS Uint64)) AS yandexuid, segment_name
    FROM (
        SELECT
            segment.id AS id,
            segment.id_type AS id_type,
            segment.segment_name AS segment_name,
            matching.yandexuid AS yandexuid
        FROM $segment_ids AS segment
        LEFT JOIN `{id_to_yandexuid_table}` AS matching
        USING (id, id_type)
    )
    WHERE yandexuid IS NOT NULL OR id_type == 'yandexuid'
)
GROUP BY yandexuid, segment_name;
"""


class MusicGenresListeners(RegularSegmentBuilder):
    name_segment_dict = {
        'music_folk_listeners': (547, 1122),
        'music_jazz_listeners': (547, 1123),
        'music_rock_listeners': (547, 1124),
        'music_classical_listeners': (547, 1125),
        'music_rap_listeners': (547, 1126),
        'music_rus_rap_listeners': (547, 1623),
        'music_electronic_listeners': (547, 1127),
        'music_dance_listeners': (547, 1128),
        'music_rnb_listeners': (547, 1129),
        'music_alternative_listeners': (547, 1130),
        'music_blues_listeners': (547, 1131),
        'music_estrada_listeners': (547, 1132),
        'music_pop_listeners': (547, 1133),
        'music_metal_listeners': (547, 1134),
        'music_reggae_listeners': (547, 1135),
        'music_indie_listeners': (547, 1136),
        'music_punk_listeners': (547, 1137),
        'music_bard_listeners': (547, 1138),
        'music_light_listeners': (547, 1139),
        'music_shanson_listeners': (547, 1140),
        'music_rock_n_roll_listeners': (547, 1624),
    }

    number_of_days = 14

    def requires(self):
        return {
            'ProcessedMusicStreamsLog': LogProcessor(
                ProcessedMusicStreamsLogForGenresListeners,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        with self.yt.TempTable() as genre_yandexuid_table:
            self.yql.query(
                join_query.format(
                    input_table=inputs['ProcessedMusicStreamsLog'].table,
                    output_table=output_path,
                    id_to_yandexuid_table=config.INDEVICE_YANDEXUID,
                    lal_sample_table=genre_yandexuid_table,
                ),
                transaction=self.transaction,
            )

            # LAL
            segment_name_to_id_dict = {
                'music_rap_listeners': 1141,
                'music_rock_listeners': 1142,
                'music_alternative_listeners': 1143,
                'music_pop_listeners': 1144,
                'music_dance_listeners': 1145,
                'music_metal_listeners': 1146,
                'music_electronic_listeners': 1147,
                'music_indie_listeners': 1148,
            }

            lals_params = list()
            for segment_name, segment_id in segment_name_to_id_dict.iteritems():
                lals_params.append(LaLParams(
                    name=segment_name,
                    id=segment_id,
                    type='lal_internal',
                    coverage=1000000,
                    include_input=False,
                ))

            self.prepare_samples_for_lal(
                input_table=genre_yandexuid_table,
                id_field='yandexuid',
                lals_params=lals_params,
            )
